// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// 葉っぱの場合と同じく、個体群をまとめた Population クラスを最初に作り、その中に現在の個体群の状態を格納する。
// サブクラス Hadani を作り、各個体はそのインスタンスとする。
// Population オブジェクトの中に配列を作り、その要素として各個体の Hadani オブジェクトを配置。

// popul_max は、このPopulationの最大累計個体数。同時に生存可能な個体数ではなく、ゲーム内で生成・操作可能な上限数。
// 現在何個体のHadaniが使用されているかは、this.cumul に格納される。新しいHadaniは == this.cumul+1 にアクセスして作成する。
// 新しいハダニインスタンスは最初に全部作っておくが、age=0で初期化。これをage=1に上げると初めて画面内に描画され、成長を開始する。

// なお現在は「交尾」がなく、メスは成虫になった直後、超自然的な力でなぜか受精しているものとしている。こわいのう
// なのでオスはオスではなく、ただのワンダラーなのだが、下手に実装すると O(n^2) で計算時間が爆発するのでやめておく。
// ============================================================================

function Population( depth, fromX, fromY, toX, toY, popul_founder, popul_max, maleratio ) {
    this.depth_from = depth; // じつは 2D context では不要だが、描画用の深度の起点。あるいはハダニ個体の通しナンバー
    this.fromX=fromX;
    this.fromY=fromY;
    this.toX=toX;
    this.toY=toY;
    this.W = toX-fromX;
    this.H = toY-fromY;
    this.popul_founder = popul_founder; // 最初のフレームで誕生するハダニの個体数。
    this.popul_max = popul_max; // ゲーム内で取り扱うハダニ延べ数の上限。
    this.maleratio = maleratio;

    this.cumul = popul_founder; // 既にステージ上に誕生している（age>=1）ハダニの延べ数。次のstate_arrayの[1:5]の和に恒等
    this.state_array = [popul_max-popul_founder, popul_founder, 0, 0, 0, 0, 0]; // 個体群サイズの最も基本的な状態変数
    // state_array は[0]: 未誕生、[6]: 死亡を含む完全なデータ。以下の諸配列は長さ5で、現在生きている個体の情報のみを格納する。
    this.eaten_now_array = [0, 0, 0, 0, 0]; // 現フレームにおける「発育ステージ毎の被食個体数」。
    this.eaten_cumul_array = [0, 0, 0, 0, 0]; // 全フレーム累計の「発育ステージ毎の被食個体数」。
    this.eaten_total = 0; // ハダニの総累計被食数。全発育ステージの和。
    this.cease_cumul_array = [0, 0, 0, 0, 0]; // 寿命で死んだ数。実はまだ作り込んでない。
    this.death_total = 0; // 自然死と捕食を含めたハダニの総死亡数の累計

    // ゲーム始まってからの各タイムステップにおける個体数を全て収録した「配列の配列」。最初は空で、動的に要素を追加。
    // 0:未登録, 1:卵, 2:幼虫, 3:静止期, 4:雌成虫, 5:オス成虫, 6:死亡
    this.dyn_array = [[], [], [], [], [], [], []];

    // ハダニ個体の初期化。最初に上限いっぱいまで作成。場所は移動可能範囲内でランダムに散らしている。雌雄はここで決めている。
    this.ary = new Array(popul_max);
    for(var i = 0; i < popul_max; i=(i+1)|0) {
        this.ary[i] = new Hadani( i+this.depth_from, this.fromX, this.fromY, this.toX, this.toY,
                                  this.fromX+Math.random()*this.W, this.fromY+Math.random()*this.H, maleratio );
    }
    for(var i = 0; i < popul_founder; i=(i+1)|0) {
        this.ary[i].age = 1; // Hadaniオブジェクトをinitするとageにゼロが代入されるので、その後founderだけ誕生を実行する事
    }
    // 蛇足。ユーザー設定として初期個体数を増減するときは、以下のinitメソッドでHadaniを再度初期化。
};


// 初期個体数の変更メソッド。タイトル画面の難易度設定で最初に投入するハダニ親の数を増減するために使う。
Population.prototype.init = function(popul_founder, popul_max) {

    this.popul_founder = popul_founder;
    this.popul_max = popul_max;
    this.cumul = popul_founder;

    // ハダニの初期配置処理。考えるのが面倒になったので、initのたびに破棄して全部作りなおす。
    this.ary = new Array(popul_max);
    for(var i = 0; i < popul_max; i=(i+1)|0) {
        this.ary[i] = new Hadani( i+this.depth_from, this.fromX, this.fromY, this.toX, this.toY,
                                  this.fromX+Math.random()*this.W, this.fromY+Math.random()*this.H, this.maleratio );
    }
    // 初期個体分は強制的にメス個体として再代入。お前がママになるんだよ
    for(var i = 0; i < popul_founder; i=(i+1)|0) {
        this.ary[i] = new Hadani( i+this.depth_from, this.fromX, this.fromY, this.toX, this.toY,
                                  this.fromX+Math.random()*this.W, this.fromY+Math.random()*this.H, 0 );
        this.ary[i].stage = 4; // 初期個体を雌成虫にする。
        this.ary[i].age = HADANI_LIFETIME_CUMUL_ARRAY[3]+1; // 初期個体の齢を「雌成虫になった直後」に書き換え
    }

    this.state_array = [0, 0, 0, 0, 0, 0, 0];    // 個体群の状態変数を更新。
    for (var i = 0; i < this.popul_max; i=(i+1)|0) {
        this.state_array[this.ary[i].stage] += 1;
    }
    for (var i = 0; i < this.dyn_array.length; i=(i+1)|0) {
        this.dyn_array[i].push(this.state_array[i]);
    }
};


// 状態更新メソッド
Population.prototype.setStatus = function(ctx, leaf, predator, magnify, chain_plant_hadani, chain_hadani_kaburi) {

    var newborn = 0; // このフレームで新規に追加される卵の数。設計がまずいとインデックス外にアクセスして止まるので注意
    this.eaten_now_array = [0, 0, 0, 0, 0]; // 現フレームでの被食個体数の情報をリセット

    // 計算負荷を鑑みて、for文は産卵済みの個体についてのみ回すようにする。
    for(var i = 0; i < Math.min(this.cumul, this.popul_max); i=(i+1)|0) {
        this.ary[i].setStatus(ctx, leaf, predator, magnify, chain_plant_hadani, chain_hadani_kaburi); // 個体状態更新

        // もし i 番目の個体が産卵可能だったら
        if (this.ary[i].readyForOviposite > 0) {
            var ind = this.cumul + newborn; // 新しい個体のid
            // ストックがある場合のみ産卵させる。
            // 蛇足：「卵塊」つまり一度に多数の卵を産む生物だと、現在のアルゴリズムは破綻しうるので注意。
            if(ind < this.popul_max) {
                this.ary[ind].age = 1; // age=0は未産卵。age=1に上げると、卵として誕生する。
                this.ary[ind]._x = this.ary[i]._x; //
                this.ary[ind]._y = this.ary[i]._y; //
                this.ary[i].ovum -= 1; // 個体が持つ残りの卵母の数をデクリメント
            }
            newborn += 1; // ストックが無くてもnewbornを上げるべきなのか？
            this.ary[i].york -= HADANI_LAYEGG_YORK; //
            this.ary[i].readyForOviposite = 0; // 産卵可能のフラグをおろす。
        }

        // 個体の更新メソッド内で predator との距離を計算して被食フラグを立てているので、
        if (this.ary[i].eaten_now > 0) {
            this.eaten_cumul_array[this.ary[i].eaten_now-1] += 1; // 被食個体数を1つ増やす。なお卵が1
            this.eaten_now_array[this.ary[i].eaten_now-1] += 1; // 被食個体数を1つ増やす
            this.ary[i].eaten_now = 0; // フラグを元に戻す
            this.eaten_total = this.eaten_total + 1;
        }
    }
    this.cumul += newborn; // 現時点での最新ハダニナンバーを上げる

    // 個体群の状態変数を更新。 0:未登録, 1:卵, 2:幼虫, 3:静止期, 4:雌成虫, 5:オス成虫, 6:死亡
    this.state_array = [0, 0, 0, 0, 0, 0, 0];
    for (var i = 0; i < this.popul_max; i=(i+1)|0) {
        this.state_array[this.ary[i].stage] += 1;
    }
    for (var i = 0; i < this.dyn_array.length; i=(i+1)|0) {
        this.dyn_array[i].push(this.state_array[i]);
    }
};


// 個体群の描画メソッド。葉っぱの外にはみ出した体の部分が描画されるので、あまりカッコよくない。
Population.prototype.drawme = function(ctx) {
    for(var i = this.popul_max-1; i >=0 ; i=(i-1)|0) {
        this.ary[i].drawme(ctx);
    }
};

// 個体群の描画メソッド。普段はこちらを使う
Population.prototype.drawClipping = function(ctx, fromX, fromY, toX, toY) {
    var MG = 1;
    var RD = 2;

    ctx.save();

    // クリッピングエリア設定。葉っぱの外にはみ出した体の部分を隠す。
    ctx.beginPath();
    ctx.moveTo(fromX+MG+RD, fromY+MG);
    ctx.lineTo(toX-MG-RD, fromY+MG);
    ctx.quadraticCurveTo(toX-MG, fromY+MG, toX-MG, fromY+MG+RD); // right top
    ctx.lineTo(toX-MG, toY-MG-RD);
    ctx.quadraticCurveTo(toX-MG, toY-MG, toX-MG-RD, toY-MG); // right bottom
    ctx.lineTo(fromX+MG+RD, toY-MG);
    ctx.quadraticCurveTo(fromX+MG, toY-MG, fromX+MG, toY-MG-RD); // left bottom
    ctx.lineTo(fromX+MG, fromY+MG+RD);
    ctx.quadraticCurveTo(fromX+MG, fromY+MG, fromX+MG+RD, fromY+MG); // left top
    ctx.clip();

    for(var i = this.popul_max-1; i >=0 ; i=(i-1)|0) {
        this.ary[i].drawme(ctx);
//        this.ary[i].drawmeEasy(ctx); // キャラクターデザインを無視して描画するお手軽メソッド
    }

    ctx.restore();
};


// ============================================================================
// ハダニクラスを定義
// 毎フレーム実行される setStatus で、母親のインスタンスが産卵フラグを立てたとき、最新idの個体のageを1に上げる。これが産卵の処理である。
// this.age <= 0 であるときは、状態更新をスキップ。つまり原点で待機状態にあり、未誕生なので描画対象としない。
// ============================================================================

var Hadani = function(i, fromX, fromY, toX, toY, where_x, where_y, MALE_PROP) {

    this.depth = i; // 個体ナンバーであり、インスタンスの描画深度でもある
    this.fromX=fromX;
    this.fromY=fromY;
    this.toX=toX;
    this.toY=toY;
    this.size = 0;
    this.angle = Math.random()*(Math.PI*2);
    this.rotate = false;
    this.magnify = 0;
    this._x = where_x;
    this._y = where_y;
    this.speed = 0;

    this.runaway = 0; // 逃避中か。最初は0で初期化
    this.forage = 0; // 採餌中か。最初は0で初期化
    this.moment_x = 0; // そのフレームで動くかどうか_x: なお左に動く場合はマイナス1になる。
    this.moment_y = 0; // そのフレームで動くかどうか_y:
    this.move_x = 0; // 毎フレームの移動量_x
    this.move_y = 0; // 毎フレームの移動量_y
    this.york = 0; // 体内にストックされている卵黄の量。雌成虫が餌を食べることでのみゲージが溜まる。
    this.readyForOviposite = 0; // 厳密には雌だけに必要なフラグ。こいつが1であれば産卵させる。
    this.eaten_now = 0; // 現フレームにおいてカブリダニに捕食されたという情報を記録する。次フレームではリセットする。

    if (Math.random() < MALE_PROP) {
        this.male = 1;
        this.ovum = 0; // オスはもちろん子を産まない
    } else {
        this.male = 0;
        this.ovum = HADANI_FECUNDITY_MAX; // 雌の体内にストックされている卵母の数。理論上の最大産卵数
    }
    this.age = 0;
    this.stage = 0;

    // 以下はデバッグ用の追加情報表示に使う。処理速度を著しく低下させるので普段はゼロにしておく。
    this.display = 0; // 個体ステータスの表示用セグメントを出すかどうか
    if (this.display > 0) {
        this.segframe = new SSD(depth_from=100, LEN_MAX=4, digit=4, where_x=this._x+10, where_y=this._y,
                                h=15, col_R=(1-this.male)*230, col_G=30, col_B=155, data=this.depth);
    }
    this.sotowaku = 0; // 個体周辺の外枠を表示するかどうか。

    // 個体の歩行軌跡を表示するかどうか。すごく重いので、デバッグ作業時だけ有効に。
    this.locus = 0;
    if (this.locus > 0) {
        this.locus_x = new Array; // ダニの軌跡を記録するための配列。長さはあえて指定しない
        this.locus_y = new Array;
        this.locus_alpha = new Array; // ダニの軌跡の有効無効を記録するための配列
    }
}


Hadani.prototype = new Character(); // Characterオブジェクトの継承

// Hadani サブクラスにおいて、継承したメソッドのオーバーライドも可能。
// http://atamoco.boy.jp/javascript/oop/inherit-override.php
// なお、this.size などは継承元クラスと重複する内容だが、その部分はこちらから改めて指定すれば上書きされる。

// ハダニの描画準備。毎フレーム実行させる。
// 食害の計算方法。各個体の現在の位置にある葉のidを計算。対応する場所のleafオブジェクトに、leaf lifeがあれば食害フラグ立てる。
// 食害中であれば相手オブジェクトの leaf lifeを減らすと共に、自分の移動速度を差し引く。
// ハダニの移動速度を、採食中、捕食者回避時、通常時で変化させる。
// 蛇足：現在は斜め方向にも動くが、縦横だけに制限すれば、マンハッタン距離の計算だけで済むので動きが軽くなるのでは？
// 食物連鎖の切断オプションを追加。ダミー変数である chain_hadani_kaburi を使ったお手軽処理

Hadani.prototype.setStatus = function(ctx, leaf, predator, magnify, chain_plant_hadani, chain_hadani_kaburi) {

    // 以下、個体の状態更新はstageが1から5までの場合だけ必要であることに注意。
    if (this.age > 0 && this.stage <= 5) {

        // 最初に捕食者との距離 dist2kaburi を計測し、通常／逃避／被食のフェーズを決定。
        // predator._x, predator._y で捕食者の座標を取得。自分の座標は this._x, this._y であるので、
        // dist2kaburi <= CRITICAL_DIST_ARRAY で被食処理
        if (chain_hadani_kaburi > 0) {
            this.dist2kaburi = Math.pow(this._x-predator._x, 2) + Math.pow(this._y-predator._y, 2);
            if( this.dist2kaburi <= Math.pow( CRITICAL_DIST_ARRAY[this.stage-1], 2 )) {
                this.eaten_now = this.stage; // 現フレームでの被食フラグを立てる。なお、ageを上げる前に実施する必要。
                this.age += 10000;
            }
        }
        // 上で計算済みの dist2kaburi <= AVOID_DIST_ARRAY ならば、まず回避処理を入れる。
        // ちなみに卵のAVOID_DISTはゼロである。「自走して逃げる卵」があったら怖いね。
        // 最初は挙動がおかしかった。カブリダニから見てハダニが第2象限にいるとき、逃げずに吸い寄せられてしまう。
        // 実は全体の座標系が第4象限なので、this.vect_k2h が (yがマイナス、xもマイナス) になっているということだった。
        // 解決策として、yをマイナスにしてから投入。
        this.vect_k2h = [ this._x-predator._x, this._y-predator._y ];
        if( this.dist2kaburi <= Math.pow( AVOID_DIST_ARRAY[this.stage-1], 2 )) {
            this.runaway = 1; // 逃避中フラグ立てる
            this.forage = 0; // 採餌は中断。実はここで判定しても意味ない？
            this.angle = Math.atan2(-this.vect_k2h[1], this.vect_k2h[0]); // 第4象限なのでyをマイナスにする必要
        } else {
            this.runaway = 0; // 逃避中フラグおろす
            this.forage = 1; // このフレームにおける採餌フラグを立て直す
        }
        // さらに個体のランダム死亡 HADANI_CEASE_PROB_ARRAY を入れるならば、ここか

        // 現在の齢を更新。なお外側の ifで、すでにthis.age > 0 (産卵済み) なのは確定
        if (this.age <= HADANI_LIFETIME_CUMUL_ARRAY[0]) {
            this.stage = 1; // 1卵 2幼虫 3静止期 4雌成虫 5オス成虫 6死亡
        } else if (this.age > HADANI_LIFETIME_CUMUL_ARRAY[5]) {
            this.stage = 6; // 1卵 2幼虫 3静止期 4雌成虫 5オス成虫 6死亡
        } else {
            for (var i = 0; i < 5; i=(i+1)|0) {
                if (HADANI_LIFETIME_CUMUL_ARRAY[i] < this.age && this.age <= HADANI_LIFETIME_CUMUL_ARRAY[i+1]) {
                    this.stage = i+1; // 1卵 2幼虫 3静止期 4雌成虫 5オス成虫 6死亡
                }
            }
            if(this.male > 0 && this.stage == 4) {
                this.age += HADANI_LEN_FEMALE;
                this.stage = 5;
            }
            if(this.male < 1 && this.stage == 5) {
                this.age += HADANI_LEN_MALE;
                this.stage = 6;
            }

            // ハダニ個体の現在位置にある葉パッチを取得し、吸汁の処理。以前のバージョンでは位置決定後に実行していたが変更。
            // 2016/05/26 葉っぱとハダニのリンクを切断するオプションを追加。リンク有無がダミー変数化されているので、乗算するだけ
            // 2016/05/27 成虫が死ぬときに下の葉っぱを消滅させるバグを修正。栄養間リンクを切っていても起こるバグであった。
            // 理由は単純で、var mf に代入すべき HADANI_FORAGE_ARRAY[this.stage-1] が undefined だったせい
            var poi = leaf.returnPatch(this._x, this._y);
            // 下のifがないと、ダニが範囲外アクセスして落ちる事がある。
            if (poi[0] >= 0 && poi[0] < leaf.nx && poi[1] >= 0 && poi[1] < leaf.ny) {
                var leaf_life = leaf.ary[poi[0]][poi[1]].life;
                if (leaf_life > 0 && this.stage > 0 && this.stage < 6) {
                    this.forage *= 1; // もともと逃避中の場合は 0 なので、ここでフラグを勝手に追加しないよう乗算代入に。
                    var mf = HADANI_FORAGE_ARRAY[this.stage-1]; // mfはハダニのフレームあたり最大食害量
                    var eiyo = mf; // eiyoは葉から吸汁された栄養の、ダニへの移行分
                    if(mf <= leaf_life) {
                        leaf.ary[poi[0]][poi[1]].life -= mf*chain_plant_hadani; // 葉の残りlifeが足りているとき
                    } else {
                        leaf.ary[poi[0]][poi[1]].life = 0; // 足りないとき。絞れるだけ絞り尽くす
                        eiyo = leaf_life;
                    }
                    // 雌成虫のみ、繁殖のための栄養ゲージを変化させる。
                    if(this.male < 1 && this.stage == 4) {
                        this.york += eiyo;
                        if(this.york >= HADANI_LAYEGG_YORK && this.ovum <= HADANI_FECUNDITY_MAX) {
                            this.readyForOviposite = 1; // 産卵処理は個体群レベルのイベント。ここでフラグを立てておく。
                        }
                    }
                    // 栄養状態と生死、移動速度などのリンクが未実装なので、食べなくても死なないという怪しいことに。
                } else {
                    this.forage *= 0; // leaf_life が足りない場合は採餌中フラグをおろす。
                }
            }

            // ここから実際の方向転換。カブリ→ハダニのベクトルは k2h(x,y) = ( this._x-predator._x, this._y-predator._y )
            // Math.atan2(y,x) でまっすぐ逃避する方向の角度（単位ラジアン）を与える。
            // 方向転換の挿入場所は被食判定より後

            this.former_angle = this.angle; // 前フレームの位置状態を退避
            this.former_x = this._x;
            this.former_y = this._y;
            this.former_move_x = this.move_x;
            this.former_move_y = this.move_y;

            this.magnify = magnify; // 毎フレームのハダニの表示倍率。これ実は不要な気がするぞ
            this.moment_x = 0; // ハダニの移動有無を毎フレーム初期化する
            this.moment_y = 0;
            this.rollback = 0; // 葉っぱからのはみ出し時のオフセットを行ったか否か

            var round_limit = HADANI_HEAD_ARRAY[this.stage-1]; // 回転角の制限単位：1で360度
            var round_rad = (Math.random()-0.5)*(Math.PI*2)*round_limit*(1+this.forage)*(2-this.runaway)/3;
            // ランダムな角度を単位ラジアンで生成する。さらに食事中は2倍、逃避中は半分の首振り角になる。
            this.angle = round_rad + this.angle;
            // ダニの角度が0以上2Pi以下に収まるよう、オフセットする。
            // これ、ランダムネスをプラス方向だけにすれば、if文を1つ減らせる。
            if (this.angle >= 2*Math.PI) {
                this.angle = this.angle % 2*Math.PI;
            } else if (this.angle < 0) {
                this.angle = (this.angle + 2*Math.PI) % 2*Math.PI;
            }

            // 移動速度は毎フレーム変わる。
            this.speed = HADANI_SPEED_ARRAY[this.stage-1]*(1.5-this.forage)*(2+this.runaway)/3.7;
            var di = Math.random()*this.speed;
            this.move_x = Math.round( Math.cos( this.angle )*di );
            this.move_y = Math.round( Math.sin( this.angle )*di );
            this._x += this.move_x; // ここで実際にキャラの位置を更新
            this._y += this.move_y;

            // ハダニが画面外に逃げないよう、ロールバック
            if (this._x >= this.toX) {
                this._x = this.fromX + ((this._x-this.fromX) % (this.toX-this.fromX));
                this.rollback = 1;
            } else if (this._x < this.fromX) {
                this._x = (this._x-this.fromX) + this.toX;
                this.rollback = 1;
            }
            if (this._y >= this.toY) {
                this._y = this.fromY + ((this._y-this.fromY) % (this.toY-this.fromY));
                this.rollback = 1;
            } else if (this._y < this.fromY) {
                this._y = (this._y-this.fromY) + this.toY;
                this.rollback = 1;
            }

            if (this.locus > 0) {
                this.locus_x.push(this._x); // 軌跡のつぎたし（必要時のみ）
                this.locus_y.push(this._y);
                this.locus_alpha.push(1-this.rollback); // ロールバック部分を透明化しないと画面が真っ黒に。
            }

            // ディスプレイ描画。num = this.depth で個体ナンバー。poi[0]で対応する葉パッチ、yorkで栄養ストック
            if (this.display > 0) {
                this.segframe.setStatus(num=this.depth, where_x=this._x, where_y=this._y);
            }
        } // ここまで、ハダニの齢が1以上5以下の場合しか必要ない処理。

        this.age += 1; // 最後にハダニの齢をインクリメント
    }
}


// 毎フレーム、自身を描画させるメソッド。
// 引数の ctx で、どのコンテキスト上に描画させるかを明示的に指定する。キャラクターデザイン依存部分はサブルーチン化した。
Hadani.prototype.drawme = function(ctx) {

    if (this.age > 1 && this.stage < 6) {

        // ハダニの移動軌跡（オプション）
        if (this.locus > 0) {
            for (var i = 0; i < this.locus_x.length-1; i=(i+1)|0) {
                ctx.beginPath();
                ctx.moveTo(this.locus_x[i], this.locus_y[i]);
                ctx.lineTo( this.locus_x[i+1], this.locus_y[i+1] );
                ctx.strokeStyle = "rgba(255, 255, 5, " + this.locus_alpha[i+1] + ")";
                ctx.stroke();
            }
        }

        if (this.stage <= 1) {
            this.drawEgg(ctx);
        } else if (this.stage <= 2) {
            this.drawLarva(ctx);
        } else if (this.stage <= 3) {
            this.drawChrysalis(ctx);
        } else if (this.male < 1 && this.stage <= 4) {
            this.drawAdult(ctx);
        } else if (this.male > 0 && this.stage <= 5) {
            this.drawMale(ctx);
        }

        // 個体ステータス（オプション）
        if (this.display > 0) {
            if (this.male > 0 && this.stage < 6) {
                this.segframe.drawme(ctx);
            }
            if (this.male < 1 && this.stage < 5) {
                this.segframe.drawme(ctx);
            }
        }
    }
}

// 以下は描画を軽くするため、キャラクターデザインを無視してただの四角を描かせている
Hadani.prototype.drawmeEasy = function(ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[4];
    var W_OBJ = 6; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 16; // キャラクター描画領域の高さ。

    if (this.age > 1 && this.stage < 6) {
        ctx.save();

        // ハダニの移動軌跡（オプション）
        if (this.locus > 0) {
            for (var i = 0; i < this.locus_x.length-1; i=(i+1)|0) {
                ctx.beginPath();
                ctx.moveTo(this.locus_x[i], this.locus_y[i]);
                ctx.lineTo( this.locus_x[i+1], this.locus_y[i+1] );
                ctx.strokeStyle = "rgba(255, 255, 5, " + this.locus_alpha[i+1] + ")";
                ctx.stroke();
            }
        }

        this.offset_norot(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ); // 計算負荷は極めて大。

        ctx.beginPath();
        ctx.moveTo(0, 0);
        ctx.lineTo(0, H_OBJ);
        ctx.lineTo(W_OBJ, H_OBJ);
        ctx.lineTo(W_OBJ, 0);
        ctx.lineTo(0, 0);
        ctx.closePath();
        ctx.strokeStyle = "rgba(5, 5, 5, 1)";
        ctx.stroke();

        // 個体ステータス（オプション）
        if (this.display > 0) {
            if (this.male > 0 && this.stage < 6) {
                this.segframe.drawme(ctx);
            }
            if (this.male < 1 && this.stage < 5) {
                this.segframe.drawme(ctx);
            }
        }

        ctx.restore();
    }
}

// ============================================================================
// キャラクターデザインの指定。
// 表示倍率 magnify の情報は、上で init したときに this.magnify へ入っているので、そいつを呼び出す。

// なお現在のコードでは、描画領域の起点は左上だが、回転は描画領域の中央を中心にしている。
// キャラクターの座標を動的に処理するには、描画領域も中央を起点にしたい。

// 2016/06/20 現在のコードでは描画そのものがクラスメソッドになっているが、これはナンセンスかも？
// ============================================================================

Hadani.prototype.drawEgg = function (ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[0];
    var W_OBJ = 16; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 16; // キャラクター描画領域の高さ。

    ctx.save();
    this.offset_norot(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ); // 回転させない
    renderHadaniEgg(ctx, HADANI_COLOR);
    if (this.sotowaku > 0) {
        renderSotowaku(ctx, W_OBJ, H_OBJ);
    }

    ctx.restore();
}


Hadani.prototype.drawLarva = function (ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[1];
    var W_OBJ = 29; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 33; // キャラクター描画領域の高さ。

    ctx.save();
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ);
    renderHadaniLarva(ctx, HADANI_COLOR);
    if (this.sotowaku > 0) {
        renderSotowaku(ctx, W_OBJ, H_OBJ);
    }

    ctx.restore();
}

Hadani.prototype.drawChrysalis = function (ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[2];
    var LEG_COLOR = HADANI_LEG_COLOR;
    var EYE_COLOR = HADANI_EYE_COLOR;
    var W_OBJ = 21; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 38; // キャラクター描画領域の高さ。

    ctx.save();
    this.offset_norot(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ); // 回転させない
    renderHadaniChrysalis(ctx, HADANI_COLOR, LEG_COLOR, EYE_COLOR);
    if (this.sotowaku > 0) {
        renderSotowaku(ctx, W_OBJ, H_OBJ);
    }

    ctx.restore();
}


Hadani.prototype.drawAdult = function (ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[3];
    var W_OBJ = 32; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 48; // キャラクター描画領域の高さ。

    ctx.save();
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ);
    renderHadaniAdult(ctx, HADANI_COLOR);
    if (this.sotowaku > 0) {
        renderSotowaku(ctx, W_OBJ, H_OBJ);
    }

    ctx.restore();
}


Hadani.prototype.drawMale = function (ctx) {

    var HADANI_COLOR = HADANI_COL_ARRAY[4];
    var W_OBJ = 28; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 45; // キャラクター描画領域の高さ。

    ctx.save();
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ);
    renderHadaniMale(ctx, HADANI_COLOR);
    if (this.sotowaku > 0) {
        renderSotowaku(ctx, W_OBJ, H_OBJ);
    }

    ctx.restore();
}

function renderSotowaku(ctx, W_OBJ, H_OBJ) {
    ctx.strokeStyle = "rgba(5, 5, 5, 1)";
    ctx.fillStyle = "rgb(0,0,0)";

    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, H_OBJ);
    ctx.lineTo(W_OBJ, H_OBJ);
    ctx.lineTo(W_OBJ, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.stroke();

    // ボディ中央マーカー
    ctx.beginPath();
    ctx.arc(W_OBJ/2, H_OBJ/2, 3, 0, Math.PI * 2, false);
    ctx.fill();
}


// ============================================================================
/*
ハダニ卵のデザインを定義。
*/
// ============================================================================

function renderHadaniEgg(ctx, cFILL) {
    var W_OBJ = 16;
    var H_OBJ = 16;
    ctx.beginPath();
    ctx.moveTo(16.0, 8.0);
    ctx.bezierCurveTo(16.0, 12.4, 12.4, 16.0, 8.0, 16.0);
    ctx.bezierCurveTo(3.6, 16.0, 0.0, 12.4, 0.0, 8.0);
    ctx.bezierCurveTo(0.0, 3.6, 3.6, 0.0, 8.0, 0.0);
    ctx.bezierCurveTo(12.4, 0.0, 16.0, 3.6, 16.0, 8.0);
    ctx.closePath();
    ctx.fillStyle = cFILL;
    ctx.fill();
};

// ============================================================================
/*
ハダニ幼虫のデザインを定義。
*/
// ============================================================================

function renderHadaniLarva(ctx, cFILL) {
    var W_OBJ = 29;
    var H_OBJ = 33;
    ctx.fillStyle = cFILL;

    // larva/legsL/
    ctx.beginPath();
    ctx.moveTo(7.0, 6.1);
    ctx.lineTo(3.6, 0.0);
    ctx.lineTo(7.8, 11.6);
    ctx.lineTo(7.0, 6.1);
    ctx.closePath();
    ctx.fill();

    // larva/legsL/
    ctx.beginPath();
    ctx.moveTo(0.0, 8.1);
    ctx.lineTo(6.9, 13.5);
    ctx.lineTo(3.8, 9.9);
    ctx.lineTo(0.0, 8.1);
    ctx.closePath();
    ctx.fill();

    // larva/legsL/
    ctx.beginPath();
    ctx.moveTo(3.6, 32.0);
    ctx.lineTo(6.3, 21.7);
    ctx.lineTo(4.5, 23.8);
    ctx.lineTo(3.6, 32.0);
    ctx.closePath();
    ctx.fill();

    // larva/legsR/
    ctx.beginPath();
    ctx.moveTo(21.3, 6.1);
    ctx.lineTo(20.5, 11.6);
    ctx.lineTo(24.7, 0.0);
    ctx.lineTo(21.3, 6.1);
    ctx.closePath();
    ctx.fill();

    // larva/legsR/
    ctx.beginPath();
    ctx.moveTo(24.5, 9.9);
    ctx.lineTo(21.4, 13.5);
    ctx.lineTo(28.3, 8.1);
    ctx.lineTo(24.5, 9.9);
    ctx.closePath();
    ctx.fill();

    // larva/legsR/
    ctx.beginPath();
    ctx.moveTo(22.0, 21.7);
    ctx.lineTo(24.7, 32.0);
    ctx.lineTo(23.7, 23.8);
    ctx.lineTo(22.0, 21.7);
    ctx.closePath();
    ctx.fill();

    // larva/body
    ctx.beginPath();
    ctx.moveTo(16.2, 7.6);
    ctx.lineTo(15.8, 5.0);
    ctx.lineTo(14.1, 2.6);
    ctx.lineTo(12.4, 5.0);
    ctx.lineTo(12.1, 7.6);
    ctx.bezierCurveTo(8.6, 8.9, 6.0, 13.5, 6.0, 18.9);
    ctx.bezierCurveTo(6.0, 25.3, 9.7, 30.5, 14.1, 30.5);
    ctx.bezierCurveTo(18.6, 30.5, 22.3, 25.3, 22.3, 18.9);
    ctx.bezierCurveTo(22.3, 13.5, 19.7, 8.9, 16.2, 7.6);
    ctx.closePath();
    ctx.fill();
};

// ============================================================================
/*
ハダニ若虫のデザインを定義。
*/
// ============================================================================

function renderHadaniChrysalis(ctx, cFILL, cLEG, cEYE) {
    // 定数の初期化
    var W_OBJ = 21;
    var H_OBJ = 38;
    var HADANI_COLOR = cFILL;
    var LEG_COLOR = cLEG;
    var EYE_COLOR = cEYE;

    var alpha = ctx.globalAlpha; // ここが新しい。
    var gradient;
    ctx.lineCap = "square"; // 線端を四角に（切り落としbuttではない）
    ctx.lineJoin = "round"; // 角を丸く
    ctx.lineWidth = 1.5; // 線幅

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(5.3, 5.8);
    ctx.lineTo(6.5, 1.3);
    ctx.strokeStyle = LEG_COLOR; // 線の色
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(1.4, 11.6);
    ctx.lineTo(3.2, 3.7);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(14.6, 5.6);
    ctx.lineTo(13.6, 0.9);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(18.6, 11.6);
    ctx.lineTo(16.8, 3.4);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(0.9, 26.1);
    ctx.lineTo(2.5, 33.8);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(4.3, 32.5);
    ctx.lineTo(4.8, 36.4);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(19.1, 26.1);
    ctx.lineTo(18.1, 33.8);
    ctx.stroke();

    // teleiochrysalis/legs/
    ctx.beginPath();
    ctx.moveTo(15.7, 32.5);
    ctx.lineTo(15.0, 36.9);
    ctx.stroke();

    // teleiochrysalis/chelicerae

    // teleiochrysalis/chelicerae/cheliceraR
    ctx.beginPath();
    ctx.moveTo(11.3, 4.0);
    ctx.lineTo(10.3, 1.6);
    ctx.strokeStyle = LEG_COLOR;
    ctx.lineCap = "round";
    ctx.lineJoin = "round";
    ctx.stroke();

    // teleiochrysalis/chelicerae/cheliceraL
    ctx.beginPath();
    ctx.moveTo(8.5, 4.0);
    ctx.lineTo(9.5, 1.6);
    ctx.stroke();

    // teleiochrysalis/body
    ctx.beginPath();
    ctx.moveTo(0.0, 19.6);
    ctx.bezierCurveTo(0.0, 11.0, 4.5, 3.9, 10.0, 3.9);
    ctx.bezierCurveTo(15.5, 3.9, 20.0, 11.0, 20.0, 19.6);
    ctx.bezierCurveTo(20.0, 28.3, 15.5, 35.4, 10.0, 35.4);
    ctx.bezierCurveTo(4.5, 35.4, 0.0, 28.3, 0.0, 19.6);
    ctx.closePath();
    ctx.fillStyle = HADANI_COLOR;
    ctx.fill();

    // teleiochrysalis/spot
    ctx.globalCompositeOperation = 'lighter'; //
    ctx.globalAlpha = alpha * 0.49;

    // teleiochrysalis/spot/
    ctx.beginPath();
    ctx.moveTo(0.8, 25.4);
    ctx.bezierCurveTo(0.5, 22.8, 1.3, 20.5, 2.7, 20.3);
    ctx.bezierCurveTo(4.1, 20.2, 5.4, 22.2, 5.7, 24.9);
    ctx.bezierCurveTo(6.0, 27.5, 5.1, 29.8, 3.8, 29.9);
    ctx.bezierCurveTo(2.4, 30.1, 1.1, 28.1, 0.8, 25.4);
    ctx.closePath();
    gradient = ctx.createLinearGradient(3.9, 20.5, 2.6, 29.8);
    gradient.addColorStop(0.00, "rgb(255, 255, 255)");
    gradient.addColorStop(1.00, "rgb(0, 0, 0)");
    ctx.fillStyle = gradient;
    ctx.fill();

    // teleiochrysalis/spot/
    ctx.beginPath();
    ctx.moveTo(19.2, 25.4);
    ctx.bezierCurveTo(19.5, 22.8, 18.7, 20.5, 17.3, 20.3);
    ctx.bezierCurveTo(16.0, 20.2, 14.6, 22.2, 14.3, 24.9);
    ctx.bezierCurveTo(14.0, 27.5, 14.9, 29.8, 16.2, 29.9);
    ctx.bezierCurveTo(17.6, 30.1, 18.9, 28.1, 19.2, 25.4);
    ctx.closePath();
    gradient = ctx.createLinearGradient(14.2, 23.6, 19.4, 26.7);
    gradient.addColorStop(0.00, "rgb(255, 255, 255)");
    gradient.addColorStop(1.00, "rgb(0, 0, 0)");
    ctx.fillStyle = gradient;
    ctx.fill();

    ctx.globalCompositeOperation = 'source-over'; // 描画モードを戻す

    // teleiochrysalis/eyes
    ctx.globalAlpha = alpha * 1.00;

    // teleiochrysalis/eyes/eyeR
    ctx.beginPath();
    ctx.moveTo(14.8, 10.7);
    ctx.lineTo(14.3, 9.6);
    ctx.lineWidth = 1.1; // 線幅
    ctx.strokeStyle = EYE_COLOR; // 線の色
    ctx.lineCap = "round";
    ctx.lineJoin = "round";
    ctx.stroke();

    // teleiochrysalis/eyes/eyeL
    ctx.beginPath();
    ctx.moveTo(5.1, 10.7);
    ctx.lineTo(5.5, 9.6);
    ctx.stroke();
};

// ============================================================================
/*
ハダニ雌成虫のデザインを定義。
*/
// ============================================================================

function renderHadaniAdult(ctx, cFILL) {
    var W_OBJ = 32;
    var H_OBJ = 48;
    ctx.fillStyle = cFILL;

    // hadaniFemale/legs/LegsR/
    ctx.beginPath();
    ctx.moveTo(20.8, 7.8);
    ctx.lineTo(20.3, 14.3);
    ctx.lineTo(24.6, 0.0);
    ctx.lineTo(20.8, 7.8);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/LegsR/
    ctx.beginPath();
    ctx.moveTo(25.2, 13.5);
    ctx.lineTo(23.7, 17.5);
    ctx.lineTo(30.7, 9.0);
    ctx.lineTo(25.2, 13.5);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/LegsR/
    ctx.beginPath();
    ctx.moveTo(29.7, 32.3);
    ctx.lineTo(27.1, 29.6);
    ctx.lineTo(31.6, 39.2);
    ctx.lineTo(29.7, 32.3);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/LegsR/
    ctx.beginPath();
    ctx.moveTo(28.0, 40.0);
    ctx.lineTo(25.6, 36.8);
    ctx.lineTo(29.5, 48.0);
    ctx.lineTo(28.0, 40.0);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/legsL/
    ctx.beginPath();
    ctx.moveTo(10.8, 7.8);
    ctx.lineTo(6.9, 0.0);
    ctx.lineTo(11.2, 14.3);
    ctx.lineTo(10.8, 7.8);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/legsL/
    ctx.beginPath();
    ctx.moveTo(6.4, 13.5);
    ctx.lineTo(0.9, 9.0);
    ctx.lineTo(7.9, 17.5);
    ctx.lineTo(6.4, 13.5);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/legsL/
    ctx.beginPath();
    ctx.moveTo(1.9, 32.3);
    ctx.lineTo(0.0, 39.2);
    ctx.lineTo(4.5, 29.6);
    ctx.lineTo(1.9, 32.3);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/legs/legsL/
    ctx.beginPath();
    ctx.moveTo(3.5, 40.0);
    ctx.lineTo(2.1, 48.0);
    ctx.lineTo(5.9, 36.8);
    ctx.lineTo(3.5, 40.0);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/chelicerae
    // hadaniFemale/chelicerae/cheliceraR
    ctx.beginPath();
    ctx.moveTo(17.9, 9.4);
    ctx.lineTo(16.2, 7.3);
    ctx.lineTo(16.2, 12.0);
    ctx.lineTo(18.6, 13.2);
    ctx.lineTo(17.9, 9.4);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/chelicerae/cheliceraL
    ctx.beginPath();
    ctx.moveTo(13.7, 9.4);
    ctx.lineTo(13.0, 13.2);
    ctx.lineTo(15.3, 12.0);
    ctx.lineTo(15.3, 7.3);
    ctx.lineTo(13.7, 9.4);
    ctx.closePath();
    ctx.fill();

    // hadaniFemale/body
    ctx.beginPath();
    ctx.moveTo(4.5, 29.0);
    ctx.bezierCurveTo(4.5, 20.1, 9.5, 12.9, 15.8, 12.9);
    ctx.bezierCurveTo(22.0, 12.9, 27.1, 20.1, 27.1, 29.0);
    ctx.bezierCurveTo(27.1, 37.8, 22.0, 45.0, 15.8, 45.0);
    ctx.lineTo(15.8, 45.0);
    ctx.bezierCurveTo(9.5, 45.0, 4.5, 37.8, 4.5, 29.0);
    ctx.closePath();
    ctx.fill();
}

// ============================================================================
/*
ハダニオスのデザインを定義。
*/
// ============================================================================

function renderHadaniMale(ctx, cFILL) {
    var W_OBJ = 28;
    var H_OBJ = 45;
    ctx.fillStyle = cFILL;

    // 1/leg/legR/
    ctx.beginPath();
    ctx.moveTo(19.5, 34.5);
    ctx.lineTo(21.1, 38.7);
    ctx.lineTo(21.4, 45.0);
    ctx.lineTo(19.5, 34.5);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legR/
    ctx.beginPath();
    ctx.moveTo(21.6, 29.9);
    ctx.lineTo(24.8, 34.3);
    ctx.lineTo(25.8, 37.6);
    ctx.lineTo(21.6, 29.9);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legR/
    ctx.beginPath();
    ctx.moveTo(21.6, 19.7);
    ctx.lineTo(25.9, 16.8);
    ctx.lineTo(27.9, 14.1);
    ctx.lineTo(21.6, 19.7);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legR/
    ctx.beginPath();
    ctx.moveTo(18.8, 15.1);
    ctx.lineTo(24.0, 8.7);
    ctx.lineTo(27.0, 3.1);
    ctx.lineTo(27.8, 0.0);
    ctx.lineTo(18.8, 15.1);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legL/
    ctx.beginPath();
    ctx.moveTo(8.4, 34.5);
    ctx.lineTo(6.8, 38.7);
    ctx.lineTo(6.5, 45.0);
    ctx.lineTo(8.4, 34.5);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legL/
    ctx.beginPath();
    ctx.moveTo(6.2, 29.9);
    ctx.lineTo(3.1, 34.3);
    ctx.lineTo(2.1, 37.6);
    ctx.lineTo(6.2, 29.9);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legL/
    ctx.beginPath();
    ctx.moveTo(6.2, 19.7);
    ctx.lineTo(1.9, 16.8);
    ctx.lineTo(0.0, 14.1);
    ctx.lineTo(6.2, 19.7);
    ctx.closePath();
    ctx.fill();

    // 1/leg/legL/
    ctx.beginPath();
    ctx.moveTo(9.0, 15.1);
    ctx.lineTo(3.9, 8.7);
    ctx.lineTo(0.9, 3.1);
    ctx.lineTo(0.1, 0.0);
    ctx.lineTo(9.0, 15.1);
    ctx.closePath();
    ctx.fill();

    // 1/body/body
    ctx.beginPath();
    ctx.moveTo(13.9, 13.0);
    ctx.lineTo(16.9, 13.6);
    ctx.lineTo(21.1, 16.8);
    ctx.lineTo(23.0, 25.5);
    ctx.lineTo(22.8, 27.4);
    ctx.lineTo(19.5, 34.5);
    ctx.lineTo(18.6, 36.6);
    ctx.lineTo(13.9, 43.5);
    ctx.lineTo(9.3, 36.6);
    ctx.lineTo(8.4, 34.5);
    ctx.lineTo(5.1, 27.4);
    ctx.lineTo(4.9, 25.5);
    ctx.lineTo(6.8, 16.8);
    ctx.lineTo(10.9, 13.6);
    ctx.lineTo(14.0, 13.0);
    ctx.fill();

    // 1/body/chelicerae/cheliceraR
    ctx.beginPath();
    ctx.moveTo(16.9, 13.6);
    ctx.lineTo(16.2, 9.4);
    ctx.lineTo(14.4, 7.1);
    ctx.lineTo(14.4, 12.2);
    ctx.lineTo(16.9, 13.6);
    ctx.closePath();
    ctx.fill();

    // 1/body/chelicerae/cheliceraL
    ctx.beginPath();
    ctx.moveTo(10.9, 13.6);
    ctx.lineTo(11.7, 9.4);
    ctx.lineTo(13.5, 7.1);
    ctx.lineTo(13.5, 12.2);
    ctx.lineTo(10.9, 13.6);
    ctx.closePath();
    ctx.fill();
}


// ============================================================================
// ハダニ個体群の初期サイズに基づき、ダニたちをタイトル画面に描画する。
// ============================================================================


function InitialEnemies(depth, fromX, fromY, toX, toY, founder_max) {
    this.depth_from = depth; // まだサポートしていないが、描画用の深度の起点。あるいはハダニ個体の通しナンバー
    this.fromX=fromX;
    this.fromY=fromY;
    this.toX=toX;
    this.toY=toY;
    this.W = toX-fromX;
    this.H = toY-fromY;
    this.founder_max = Math.floor(founder_max);

    // ハダニの初期配置処理。最初に上限いっぱいまで作成。
    this.ary = new Array(founder_max);
    for(var i = 0; i < founder_max; i++) {
        this.ary[i] = new InitialHadani( this.fromX+(i/(founder_max-1))*this.W,
                                         this.fromY+((i%2)*founder_max/(founder_max-1))*this.H );
    }
};


// 状態更新メソッド
InitialEnemies.prototype.setStatus = function(magnify, x) {
    if (x <= 0) {
        for(var i = 0; i < this.founder_max; i++) {
            this.ary[i].setStatus(magnify, 0);
        }
    } else if (x >= this.popul_max) {
        for(var i = 0; i < this.founder_max; i++) {
            this.ary[i].setStatus(magnify, 1);
        }
    } else {
        for(var i = 0; i < Math.floor(x); i++) {
            this.ary[i].setStatus(magnify, 1);
        }
        for(var i = Math.floor(x); i < this.founder_max; i++) {
            this.ary[i].setStatus(magnify, 0);
        }
    }
};


// 描画メソッド
InitialEnemies.prototype.drawme = function(ctx) {
    for(var i = (this.founder_max - 1); i >=0; i--) {
        this.ary[i].drawme(ctx);
    }
};


var InitialHadani = function(where_x, where_y, angle, magnify) {
    this.size = 0;
    this.angle = angle;
    this.magnify = magnify;
    this._x = where_x;
    this._y = where_y;
    this.used = 0; // 初期個体数にカウントするかどうか。
}

InitialHadani.prototype.setStatus = function(magnify, x) {
    this.magnify = magnify;
    this.used = x;
}

InitialHadani.prototype.drawme = function(ctx) {
        if (this.used >= 1) {
            this.drawUsed(ctx);
        } else {
            this.drawNot(ctx);
        }
}

InitialHadani.prototype.offset = function(ctx, rad, magnify, where_x, where_y, W_CTX, H_CTX) {
    ctx.translate(where_x - W_CTX*magnify/2, where_y - H_CTX*magnify/2);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.translate(W_CTX/2, H_CTX/2);
    ctx.rotate(rad); // ラジアンである事に注意。
    ctx.translate(-W_CTX/2, -H_CTX/2);
}

InitialHadani.prototype.drawUsed = function (ctx) {

    var HADANI_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var W_OBJ = 32; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 48; // キャラクター描画領域の高さ。

    ctx.save(); // 事前セーブ
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ);
    renderHadaniAdult(ctx, HADANI_FILL);
    ctx.restore();
}

InitialHadani.prototype.drawNot = function (ctx) {

    var HADANI_FILL = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)"; // 塗りつぶし。
    var W_OBJ = 32; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存する。
    var H_OBJ = 48; // キャラクター描画領域の高さ。

    ctx.save(); // 事前セーブ
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_OBJ, H_OBJ);
    renderHadaniAdult(ctx, HADANI_FILL);
    ctx.restore();
}


// ============================================================================
// ハダニの各齢に対応するイラストを並べて表示する汎用関数。なお回転中心は、左端のハダニからの相対位置で定義するようアレンジ。
// たとえばハダニ5匹を intervalX=40, intervalY=0 で並べた時、アレイの中心＝回転中心ならばcx=40*2=80, cy=0 である。
// ============================================================================

function renderHadaniLineup(ctx, angle, magnify, fromX, fromY, intervalX, intervalY, cx, cy) {

    var INT_X = intervalX;
    var INT_Y = intervalY;

    ctx.save();

    ctx.translate(fromX+cx, fromY+cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(-cx, -cy);

    var W_OBJ = 16;
    var H_OBJ = 16;
    ctx.translate(-W_OBJ/2, -H_OBJ/2);
    renderHadaniEgg(ctx, HADANI_COL_ARRAY[0]);
    ctx.translate(W_OBJ/2, H_OBJ/2);

    var W_OBJ = 29;
    var H_OBJ = 33;
    ctx.translate(INT_X-W_OBJ/2, INT_Y-H_OBJ/2);
    renderHadaniLarva(ctx, HADANI_COL_ARRAY[1]);
    ctx.translate(W_OBJ/2, H_OBJ/2);

    var W_OBJ = 21;
    var H_OBJ = 38;
    ctx.translate(INT_X-W_OBJ/2, INT_Y-H_OBJ/2);
    renderHadaniChrysalis(ctx, HADANI_COL_ARRAY[2], HADANI_LEG_COLOR, HADANI_EYE_COLOR);
    ctx.translate(W_OBJ/2, H_OBJ/2);

    var W_OBJ = 32;
    var H_OBJ = 48;
    ctx.translate(INT_X-W_OBJ/2, INT_Y-H_OBJ/2);
    renderHadaniAdult(ctx, HADANI_COL_ARRAY[3]);
    ctx.translate(W_OBJ/2, H_OBJ/2);

    var W_OBJ = 28;
    var H_OBJ = 45;
    ctx.translate(INT_X-W_OBJ/2, INT_Y-H_OBJ/2);
    renderHadaniMale(ctx, HADANI_COL_ARRAY[4]);
    ctx.translate(W_OBJ/2, H_OBJ/2);

    ctx.restore();
}
