// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
/*
character.jsのファイルですでにCharacterクラスを定義している。
なので、サブクラスKaburiを定義させる。

2016/05/01
カブリダニの経験値が実装されていなかったので、パラメータを足す。
今まで食べたダニの数に応じて経験値
経験値に応じて歩行スピードが上昇するとともに、捕食後の待機時間が短くなる（正確には「速く減る」）。

カブリダニの自動歩行、自動増殖は今はまだやらない。
理由：ハダニが自動でカブリダニを避ける処理は、今のところカブリダニが1個体しかいないため、N_hadani だけ繰り返し処理を行えば良い。
今後カブリダニがハダニを追いかける処理を入れるならば、追跡対象が N_hadani だけ存在するため、
個体間距離を計算して最も近いハダニを追跡するルーチンでリソースを消費することになり、現実的でない。

さらにカブリダニが複数（N_kaburi）存在する場合、ハダニ側の回避処理も最低限 O(N_hadani^2) の計算量になってしまう。
ハダニの各個体からカブリダニ各個体への位置関係を記述したベクトルの集合があって、ハダニが次にどの角度へ動けば
次のフレームにおける最短ベクトルの長さが最長となるか。
各フレームで最も近いカブリダニから単純に遠ざかるという手もあるが、この場合2番目に近いカブリダニに近づいてしまうケースがある。

*/
// ============================================================================

var Kaburi = function(fromX, fromY, toX, toY, where_x, where_y, angle, magnify) {
    this.fromX=fromX;
    this.fromY=fromY;
    this.toX=toX;
    this.toY=toY;
    this._x = this._x_0 = where_x;
    this._y = this._y_0 = where_y;
    this.angle = this.angle_0 = -angle-Math.PI/2; // 標準だと時計回りな上、角度ゼロで初期化すると右向きであることに注意。
    this.magnify = this.magnify_0 = magnify;

    this.experience = this.EXP_0 = 0; // 移動速度と経験値
    this.speed = this.SPD_0 = 3;
    this.SPD_MAX = 10; //
    this.life = this.life_0 = 15000; // カブリダニの体力
    this.LIFE_MAX = this.life // 最大値以上には回復しない。
    this.LIFE_DEC = 15; // 時間経過に伴う体力ゲージ減少速度
    this.wait = 0; // カブリダニの移動可否（食事中は停止）
    this.WAIT_MAX = 100; // ゲームのもっさりを防ぐため、待ち時間の最大値を設定
    this.wdec = 0; // 時間経過に伴う待ちゲージ減少速度
    this.WDEC_0 = 1; // 時間経過に伴う待ちゲージ減少速度の初期値
    this.WDEC_MAX = 10; // 時間経過に伴う待ちゲージ減少速度の最大値（経験を積んだ個体の）

    this.EXPERIENCE_ARY = [1, 2, 3, 4, 5] // ハダニを食べた時の経験値の上昇量。卵、幼虫、静止期、メス、オス
    this.LIFE_ARY = [200, 400, 500, 1000, 800] // ハダニを食べた時の体力の上昇量
    this.WAIT_ARY = [20, 20, 30, 50, 30] // ハダニを食べたあとの静止時間加算量（フレーム）

    this.powersupply = 0; // 2016/06/25追加。カブリダニに動力接続して電池切れを防ぐか
    this.guideline = 0; // 2016/06/28追加。進行方向を示す補助線
}


// Characterオブジェクトの継承
Kaburi.prototype = new Character();


// カブリダニの描画準備。setStatus から 移動機能をはぶき、回転のみ実行する
Kaburi.prototype.setRotate = function(keyobj, pointer, Hadani, magnify, kaburi_power_supply) {

    this.powersupply = kaburi_power_supply;
    this.former_angle = this.angle; // カブリダニの前フレームの状態を退避
    this.former_x = this._x;
    this.former_y = this._y;

    this.magnify = magnify;
    this.rotate = false; // カブリダニが現フレームにおいて方向転換するかのフラグ。毎フレーム初期化する。
    this.moment_x = 0;
    this.moment_y = 0;
    this.moment_xy = 0;

    // キーボードの状態を取得して、カブリダニの座標を更新するための移動速度ベクトル作成。
    if (keyobj.isDown(37) == true) {
        this.moment_x = -1;    // 左キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(39) == true){
        this.moment_x = 1;    // 右キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(38) == true) {
        this.moment_y = -1;    // 上キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(40) == true) {
        this.moment_y = 1;    // 下キーが押された
        this.rotate = true;
    }

    // カブリダニの方向転換処理。arctanで、原点からのデカルト座標を角度に変換できる
    if (this.rotate==true) {
        var rad = Math.atan2(this.moment_y, this.moment_x); // アークタンジェント。yが第一引数。
        this.angle = rad;
    }
    
    // 以下は、ポインターオブジェクトからの入力で上書きする操作
    if (pointer.rho > 0.1) {
        this.angle = pointer.psi;
    }
}


// カブリダニの描画準備。毎フレーム実行させる。Hadaniは個体ではなくてPopulationオブジェクトで指定。
Kaburi.prototype.setStatus = function(keyobj, pointer, Hadani, magnify, kaburi_power_supply) {

    this.powersupply = kaburi_power_supply;
    this.former_angle = this.angle; // カブリダニの前フレームの状態を退避
    this.former_x = this._x;
    this.former_y = this._y;
    this.wait_raw = 0; // 捕食による待ち時間加算量を一時的に格納する変数
    this.magnify = magnify;
    this.rotate = false; // カブリダニが現フレームにおいて方向転換するかのフラグ。毎フレーム初期化する。
    this.moment_x = 0;
    this.moment_y = 0;
    this.moment_xy = 0;
    
    // 捕食後の待ち時間を自動的に減らす。最初は当フレーム最後だったが、円グラフ表示が狂うので次フレーム初期に変更した。
    this.wdec = Math.round(this.WDEC_MAX/(1+((this.WDEC_MAX/this.WDEC_0)-1)*Math.exp(-0.065*this.experience)));
    this.wait -= this.wdec;

    // ここに、ハダニの被食イベントを受け取って経験値・体力・静止時間を増減する処理。

    // 被食イベントについては、ハダニのPopulationクラスオブジェクトに「現フレームの齢別被食個体数」の配列を埋め込んでおき、
    // setStatusの引数にハダニPopulationを指定するのが良いと思う。
    // もちろん、ハダニの各フレーム状態更新が終了してからカブリダニの状態更新を開始する必要がある。
    // この方式の利点はカブリダニの状態更新関数内にハダニ個体数分のforループを埋め込む手間が1回分省けること。
    for (var i=0; i<5; i++) {
        this.experience += Hadani.eaten_now_array[i]*this.EXPERIENCE_ARY[i];
        this.wait_raw += Hadani.eaten_now_array[i]*this.WAIT_ARY[i];
        this.life += Hadani.eaten_now_array[i]*this.LIFE_ARY[i];
    }
    // ロジスティック式によるカブリダニの移動速度決定。Math.exp(x)はe^xです。
    this.speed = Math.round(this.SPD_MAX/(1+((this.SPD_MAX/this.SPD_0)-1)*Math.exp(-0.09*this.experience)));
    this.wait = Math.max(Math.min(this.wait+this.wait_raw, this.WAIT_MAX), 0); // 待ち時間の決定
    console.log("KABURI Exp:speed:wait (init) = "+this.experience+":"+this.speed+":"+this.wait+
                                                " ("+this.EXP_0+":"+this.SPD_0+":"+this.EXP_0+")");

    // ここからは動作の処理。life がゼロより大の場合にのみ動ける。
    // 以前は待ち時間中の方向転換も禁止していたが、スマホでのプレイ時に静止解除直後の進行方向を制御するために解禁した。
    if (this.life > 0) {

        // キーボードの状態を取得して、カブリダニの座標を更新するための移動速度ベクトル作成。
        if (keyobj.isDown(37) == true) {
            this.moment_x = -1; // 左キーが押された
            this.rotate = true;
        }
        if (keyobj.isDown(39) == true){
            this.moment_x = 1; // 右キーが押された
            this.rotate = true;
        }
        if (keyobj.isDown(38) == true) {
            this.moment_y = -1; // 上キーが押された
            this.rotate = true;
        }
        if (keyobj.isDown(40) == true) {
            this.moment_y = 1; // 下キーが押された
            this.rotate = true;
        }

        // カブリダニの方向転換処理。arctanで、原点からのデカルト座標を角度に変換できる
        if (this.rotate==true) {
            var rad = Math.atan2(this.moment_y, this.moment_x); // アークタンジェント。yが第一引数。
            this.angle = rad;
        }

        if (this.wait <= 0) {
            // カブリダニの移動速度は、斜めに動く際の移動量を三角関数で減らす必要がある。
            // 縦横いずれかの方向に動く時に0、縦横両方向に動くときに1になる元の値を、1または1.414に変換する。
            this.moment_xy = 0.414*Math.floor( (this.moment_x*this.moment_x + this.moment_y*this.moment_y)/2 ) + 1;
            this.move_x = Math.round(this.speed*this.moment_x/this.moment_xy);
            this.move_y = Math.round(this.speed*this.moment_y/this.moment_xy);
    
            // 以下は、ポインターオブジェクトからの入力で上書きする操作
            if (pointer.rho > 0.01) {
                this.angle = pointer.psi;
                this.moment_x = Math.cos(pointer.psi);
                this.moment_y = Math.sin(pointer.psi);
                this.move_x = Math.round(this.moment_x*this.speed);
                this.move_y = Math.round(this.moment_y*this.speed);
            }
            if (this.moment_x*this.moment_x+this.moment_y*this.moment_y > 0) {
                this.guideline = 1;
            } else {
                this.guideline = 0;
            }
    
            // ここで実際にキャラの位置を更新
            this._x += this.move_x;
            this._y += this.move_y;
    
            // 画面外に逃げないよう、ロールバック
            if (this._x >= this.toX) {
                this._x = this.fromX + ((this._x-this.fromX)%(this.toX-this.fromX));
            } else if (this._x < this.fromX) {
                this._x = (this._x-this.fromX) + this.toX;
            }
            if (this._y >= this.toY) {
                this._y = this.fromY + ((this._y-this.fromY)%(this.toY-this.fromY));
            } else if (this._y < this.fromY) {
                this._y = (this._y-this.fromY) + this.toY;
            }
        }
    }

    if (this.life > 0 && this.powersupply <= 0) {
        this.life = Math.max(this.life-this.LIFE_DEC, 0);
    }
    if (this.life > this.LIFE_MAX) {
        this.life = this.LIFE_MAX;
    }
}


// カブリダニの状態をリセットしてしまうメソッド。リセット対象の項目は指定できる。
Kaburi.prototype.reset = function (where, angle, magnify, experience, speed, life) {
    if (where == true) {
        this._x = this._x_0;
        this._y = this._y_0;
    }
    if (angle == true) {
        this.angle = this.angle_0;
    }
    if (magnify == true) {
        this.magnify = this.magnify_0;
    }
    if (experience == true) {
        this.experinece = this.EXP_0;
    }
    if (speed == true) {
        this.speed = this.SPD_0;
    }
    if (life == true) {
        this.life = this.life_0;
    }
    this.wait = 0;
}


// 毎フレーム、キャラクター自身を描画させるメソッド。
// なお、どのコンテキストで自身を描画させるかは自動的には決まらない。基本的に第一引数の ctx を明示的に指定する。
// angle, magnify, _x, _y の情報は、上で setStatus したときにオブジェクト内に格納されたものを呼び出す。

Kaburi.prototype.drawme = function (ctx) {
    // 定数の初期化
    var KABURI_COLOR = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]*0+"%, "+ctx.liteCol[2]+"%, 1)";
    var W = 41; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存するという問題が残っている。
    var H = 60; // キャラクター描画領域の高さ。

    ctx.save();

    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W, H);
    renderKaburiShape(ctx, KABURI_COLOR);

    ctx.restore();
}


// 毎フレーム、キャラクター自身を描画させるメソッド。
// なお以下の関数は、カブリダニが葉っぱの外（ゲームの基盤）にはみ出て描画される事を防ぐ

Kaburi.prototype.drawClipping = function (ctx, fromX, fromY, toX, toY) {
    // 定数の初期化
    var KABURI_COLOR = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]*0+"%, "+ctx.liteCol[2]+"%, 1)";
    var W = 41; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存するという問題が残っている。
    var H = 60; // キャラクター描画領域の高さ。
    var MG = 0;
    var RD = 0;

    ctx.save(); // 事前セーブ

//    ctx.shadowBlur = 0;
//    ctx.shadowColor = "rgba(0,0,0,0)";

    // クリッピングエリア
    ctx.beginPath();
    ctx.moveTo(fromX+MG+RD, fromY+MG);
    ctx.lineTo(toX-MG-RD, fromY+MG);
    ctx.quadraticCurveTo(toX-MG, fromY+MG, toX-MG, fromY+MG+RD); // right top
    ctx.lineTo(toX-MG, toY-MG-RD);
    ctx.quadraticCurveTo(toX-MG, toY-MG, toX-MG-RD, toY-MG); // right bottom
    ctx.lineTo(fromX+MG       +RD, toY-MG);
    ctx.quadraticCurveTo(fromX+MG, toY-MG, fromX+MG, toY-MG-RD); // left bottom
    ctx.lineTo(fromX+MG              , fromY+MG+RD);
    ctx.quadraticCurveTo(fromX+MG, fromY+MG, fromX+MG+RD, fromY+MG); // left top
    ctx.clip();

    // グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセットして回転を掛ける。
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W, H);

    // 補助線
    if (stage == 1 && this.guideline > 0) {
        ctx.beginPath();
        ctx.moveTo(W/2, H/2);
        ctx.lineTo(W/2, -800);
        ctx.strokeStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]*0+"%, "+ctx.liteCol[2]+"%, 1)";
        ctx.strokeWidth = 0.25;
        ctx.stroke();
        
    }
    
    ctx.shadowBlur = 2.5;
    ctx.shadowColor = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]*0.5 +"%, "+ctx.darkCol[2]+"%, 0.51)";

    renderKaburiShape(ctx, KABURI_COLOR);

    ctx.shadowBlur = 0;
    ctx.shadowColor = "rgba(0, 0, 50, 0)";

    // 残存体力を電池で。maxは15000, セグメントを5つ設けて3000ごとに切る。
    ctx.beginPath();
    ctx.moveTo(W*0.44, H*0.47);
    ctx.lineTo(W*0.44, H*0.506);
    ctx.lineTo(W*0.56, H*0.505);
    ctx.lineTo(W*0.56, H*0.47);
    ctx.closePath();
    ctx.fillStyle ="hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]*0 +"%, "+ctx.baseCol[2]+"%, 1)";
    ctx.fill();

    ctx.beginPath();
    ctx.moveTo(W*0.4, H*0.52);
    ctx.lineTo(W*0.4, H*0.82);
    ctx.lineTo(W*0.6, H*0.82);
    ctx.lineTo(W*0.6, H*0.52);
    ctx.closePath();
    ctx.strokeStyle = "rgba(85,85,85,1)";
    ctx.lineWidth = 1.5;
    ctx.lineJoin = "round";
    ctx.stroke();
    ctx.fillStyle ="hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 0.5)";
    ctx.fill();

    if (this.powersupply <= 0) {

        this.life_scale = this.life/this.LIFE_MAX;
        ctx.beginPath();
        ctx.moveTo(W*0.42, H*0.81-H*0.28*this.life_scale);
        ctx.lineTo(W*0.42, H*0.81);
        ctx.lineTo(W*0.58, H*0.81);
        ctx.lineTo(W*0.58, H*0.81-H*0.28*this.life_scale);
        ctx.closePath();
        if (this.life_scale >= 0.75) {
            ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
        } else if (this.life_scale >= 0.5) {
            ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
        } else if (this.life_scale >= 0.25) {
            ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]*0 +"%, "+ctx.semiCol[2]+"%, 1)";
        } else {
            ctx.fillStyle = "rgba(255, 45, 0, 1)";
        }
        ctx.fill();

        /*
        // 残存バッテリーに25%ずつの刻みをつける
        ctx.strokeStyle ="hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
        ctx.lineWidth = 0.750;
        ctx.lineJoin = "square";
        ctx.beginPath();
        ctx.moveTo(W*0.42, H*(0.81-0.28*0.25) );
        ctx.lineTo(W*0.58, H*(0.81-0.28*0.24) );
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(W*0.42, H*(0.81-0.28*0.5) );
        ctx.lineTo(W*0.58, H*(0.81-0.28*0.49) );
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(W*0.42, H*(0.81-0.28*0.75) );
        ctx.lineTo(W*0.58, H*(0.81-0.28*0.75) );
        ctx.stroke();
        */
    } else {
         // カブリダニが電源に接続されました
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
        ctx.strokeStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
        ctx.beginPath();
        ctx.moveTo(W*0.52, H*0.76);
        ctx.lineTo(W*0.52, H*0.71);
        ctx.lineTo(W*0.553, H*0.682);
        ctx.lineTo(W*0.56, H*0.63);
        ctx.lineTo(W*0.5, H*0.64);
        ctx.lineTo(W*0.44, H*0.63);
        ctx.lineTo(W*0.447, H*0.682);
        ctx.lineTo(W*0.48, H*0.71);
        ctx.lineTo(W*0.48, H*0.76);
        ctx.closePath();
        ctx.fill();

        ctx.lineCap = "butt";
        ctx.lineWidth = 1.0;
        ctx.beginPath();
        ctx.moveTo(W*0.525, H*0.59);
        ctx.lineTo(W*0.525, H*0.63);
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(W*0.475, H*0.59);
        ctx.lineTo(W*0.475, H*0.63);
        ctx.stroke();

        // これは電源じゃない。間違って作ってしまった
        /*
        ctx.fillStyle = "rgba(5, 5, 0, 1)";
        ctx.beginPath();
        ad = 0.015;
        ctx.moveTo(W*0.5,H*(0.67-ad));
        ctx.lineTo(W*0.5,H*0.57);
        ctx.lineTo(W*0.44,H*(0.67+ad));
        ctx.lineTo(W*0.5,H*(0.67+ad));
        ctx.lineTo(W*0.5,H*0.77);
        ctx.lineTo(W*0.56,H*(0.67-ad));
        ctx.closePath();
        ctx.fill();
        */
    }

    // 捕食後の待ち時間
    this.wait_scale = (this.wait/this.WAIT_MAX) * Math.PI*2;
    ctx.beginPath();
    ctx.arc( x = W/2, y = H*0.55, radius = H*0.7,
             startAngle = -this.angle, endAngle = this.wait_scale-this.angle, anticlockwise = false);
    ctx.arc( x = W/2, y = H*0.55, radius = H*0.6,
             startAngle = this.wait_scale-this.angle, endAngle = -this.angle, anticlockwise = true);
    ctx.closePath();
    ctx.fillStyle = KABURI_COLOR;
    ctx.fill();

    // 事後ロード。これで拡大、オフセット、回転は全てリセットされ、後続の描画処理には影響しない。
    ctx.restore();
}


// ============================================================================
/*
カブリダニの形状だけを定義。Kaburiクラスと独立に置くことで、jsのどこからでも触れるように。
*/
// ============================================================================

function renderKaburiShape(ctx, cFILL) {
    var W = 41; // キャラクター描画領域の幅。このサイズは、元のキャラクターデザインに依存するという問題が残っている。
    var H = 60; // キャラクター描画領域の高さ。
    ctx.fillStyle = cFILL;
    ctx.strokeStyle ="hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(31.9, 43.3);
    ctx.lineTo(40.0, 60.0);
    ctx.lineTo(35.9, 47.0);
    ctx.lineTo(31.9, 43.3);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(30.6, 34.5);
    ctx.lineTo(40.8, 41.0);
    ctx.lineTo(34.2, 34.5);
    ctx.lineTo(30.6, 34.5);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(28.5, 29.0);
    ctx.lineTo(35.7, 15.8);
    ctx.lineTo(30.1, 23.0);
    ctx.lineTo(28.5, 29.0);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(24.2, 23.0);
    ctx.lineTo(30.7, 0.0);
    ctx.lineTo(25.8, 12.2);
    ctx.lineTo(24.2, 23.0);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(0.0, 41.0);
    ctx.lineTo(10.2, 34.5);
    ctx.lineTo(6.7, 34.5);
    ctx.lineTo(0.0, 41.0);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(0.8, 60.0);
    ctx.lineTo(9.0, 43.3);
    ctx.lineTo(4.9, 47.0);
    ctx.lineTo(0.8, 60.0);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(12.4, 29.0);
    ctx.lineTo(10.8, 23.0);
    ctx.lineTo(5.1, 15.8);
    ctx.lineTo(12.4, 29.0);
    ctx.closePath();
    ctx.fill();

    // 1/legs/
    ctx.beginPath();
    ctx.moveTo(16.5, 23.0);
    ctx.lineTo(15.0, 12.8);
    ctx.lineTo(9.3, 0.1);
    ctx.lineTo(16.5, 23.0);
    ctx.closePath();
    ctx.fill();

    // 1/antenna

    // 1/antenna/right antenna
    ctx.beginPath();
    ctx.moveTo(21.8, 7.0);
    ctx.lineTo(21.8, 22.4);
    ctx.lineTo(23.1, 15.8);
    ctx.lineTo(21.8, 7.0);
    ctx.closePath();
    ctx.fill();

    // 1/antenna/left antenna
    ctx.beginPath();
    ctx.moveTo(17.8, 15.8);
    ctx.lineTo(19.1, 22.4);
    ctx.lineTo(19.1, 7.0);
    ctx.lineTo(17.8, 15.8);
    ctx.closePath();
    ctx.fill();

    // 1/body
    ctx.beginPath();
    ctx.moveTo(20.3, 58.1);
    ctx.bezierCurveTo(20.3, 58.1, 18.8, 57.7, 16.9, 57.2);
    ctx.bezierCurveTo(15.0, 56.8, 12.4, 55.0, 11.0, 53.2);
    ctx.bezierCurveTo(9.6, 51.5, 8.6, 47.6, 8.8, 44.4);
    ctx.lineTo(8.9, 43.2);
    ctx.bezierCurveTo(9.1, 40.1, 9.9, 35.4, 10.6, 32.7);
    ctx.bezierCurveTo(11.4, 30.1, 13.2, 26.7, 14.6, 25.1);
    ctx.bezierCurveTo(16.1, 23.6, 18.0, 21.1, 18.8, 19.7);
    ctx.bezierCurveTo(19.6, 18.2, 20.3, 17.0, 20.4, 17.0);
    ctx.bezierCurveTo(20.4, 17.0, 21.1, 18.2, 22.0, 19.7);
    ctx.bezierCurveTo(22.8, 21.2, 24.7, 23.6, 26.1, 25.2);
    ctx.bezierCurveTo(27.6, 26.7, 29.4, 30.1, 30.1, 32.8);
    ctx.bezierCurveTo(30.9, 35.4, 31.7, 40.1, 31.8, 43.2);
    ctx.lineTo(31.9, 44.5);
    ctx.bezierCurveTo(32.1, 47.6, 31.1, 51.5, 29.8, 53.3);
    ctx.bezierCurveTo(28.4, 55.0, 25.7, 56.8, 23.8, 57.3);
    ctx.bezierCurveTo(22.0, 57.7, 20.4, 58.1, 20.4, 58.1);
    ctx.closePath();
    ctx.fill();

/*
    // 1/body solid
    var W = 41;
    var H = 60;
    ctx.beginPath();
    ctx.moveTo(W*0.5, 58.1);
    ctx.lineTo(W*0.412, 57.2);
    ctx.lineTo(W*0.268, 53.2);
    ctx.lineTo(W*0.21, 44.4);
    ctx.lineTo(W*0.22, 43.2);
    ctx.lineTo(W*0.26, 32.7);
    ctx.lineTo(W*0.356, 25.1);
    ctx.lineTo(W*0.46, 19.7);
    ctx.lineTo(W*0.5, 17.0);
    ctx.lineTo(W*(1-0.46), 19.7);
    ctx.lineTo(W*(1-0.36), 25.1);
    ctx.lineTo(W*(1-0.26), 32.7);
    ctx.lineTo(W*(1-0.22), 43.2);
    ctx.lineTo(W*(1-0.21), 44.4);
    ctx.lineTo(W*(1-0.27), 53.2);
    ctx.lineTo(W*(1-0.41), 57.2);
    ctx.closePath();
*/

}
