// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// このファイルでは、タイトル画面（＋ゲームオーバーなども？）などに表示させる、主として静的なオブジェクトの形状デザインを定義。
// Main00などの各ステージ基盤を描画させるためのルーチンは gamebase.js に移動した
// ============================================================================

function IlstPhone(depth, fromX, fromY, W, H, angle, magnify, finger) {
    this.depth = depth;
    this._x = fromX; // 定数。本来はconstを付けたいが、IE10以前で非対応なので諦める
    this._y = fromY; //
    this.W = W;
    this.H = H;
    this.angle = angle;
    this.magnify = magnify;
    this.isDown32 = 0;
    this.isDown37 = 0;
    this.isDown38 = 0;
    this.isDown39 = 0;
    this.isDown40 = 0;
    this.finger = finger; // 1なら操作者の指を描画

    this.font_family='font1';
    this.font_size = 40;
    this.size_unit = "px";
};


IlstPhone.prototype.setStatus = function (keys, angle, magnify, scale11, skew12, skew21, scale22, dx, dy) {
    this.keys = keys;
    // left: 37, up: 38, right: 39, down: 40,
    // spacebar: 32, pageup: 33, pagedown: 34, end: 35, home: 36
    this.angle = angle;
    this.magnify = magnify;
    this.isDown32 = AsNumeric(this.keys.isDown(32));
    this.isDown37 = AsNumeric(this.keys.isDown(37));
    this.isDown38 = AsNumeric(this.keys.isDown(38));
    this.isDown39 = AsNumeric(this.keys.isDown(39));
    this.isDown40 = AsNumeric(this.keys.isDown(40));
    this.scale11 = scale11;
    this.skew12 = skew12; // 変形無しは0。skew12 == 1 で45度だけ右肩下がりの平行四辺形になる。
    this.skew21 = skew21; // 変形無しは0。skew21 == 1 で45度だけ下縁側が右に流れた平行四辺形になる。
    this.scale22 = scale22;
    this.translate_x = dx;
    this.translate_y = dy;
}
// いわゆる1次変換。左上の原点を中心にtheta（ラジアン）だけ半時計回りに回転する際は、
// (scale11, skew12, skew21, scale22) = ( Math.cos(theta), -Math.sin(theta), Math.sin(theta) Math.cos(theta) )
// skew12がマイナスサインでskew21がプラスサイン。ほかはプラスコサイン。


AsNumeric = function(x) {
    var ans=0;
    if (x==true) {
        ans=1;
    }
    return ans;
}

// 移動機能をはぶき、回転のみ実行する
IlstPhone.prototype.setRotate = function(keyobj) {

    this.former_angle = this.angle; // カブリダニの前フレームの状態を退避
    this.former_x = this._x; //
    this.former_y = this._y; //

    this.rotate = false; // カブリダニが現フレームにおいて方向転換するかのフラグ。毎フレーム初期化する。
    this.moment_x = 0;
    this.moment_y = 0;
    this.moment_xy = 0;

    // キーボードの状態を取得して、カブリダニの座標を更新するための移動速度ベクトル作成。
    if (keyobj.isDown(37) == true) {
        this.moment_x = -1;    // 左キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(39) == true){
        this.moment_x = 1;    // 右キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(38) == true) {
        this.moment_y = -1;    // 上キーが押された
        this.rotate = true;
    }
    if (keyobj.isDown(40) == true) {
        this.moment_y = 1;    // 下キーが押された
        this.rotate = true;
    }

    // カブリダニの方向転換処理。arctanで、原点からのデカルト座標を角度に変換できる
    if (this.rotate==true) {
        var rad = Math.atan2(this.moment_y, this.moment_x); // アークタンジェント。yが第一引数。
        this.angle = -rad-Math.PI/2;
    }
}

// グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセット。
// rotateだけだと「左上の原点を中心に、反時計回りで」という直感に反した動きなので注意。
IlstPhone.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};

// 毎フレーム、自身を描画させるメソッド。
IlstPhone.prototype.drawme = function (ctx) {

    var FILL = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)"; // 塗りつぶし。
    var STROKE = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var DISP_FILL = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)"; // 塗りつぶし。
    var MG = 1;
    var RD = this.W/12;

    ctx.save(); // 事前セーブ
    this.offset(ctx, this.angle, this.magnify, this._x-this.W/2, this._y-this.H/2, this.W, this.H, this._x+this.W/2, this._y+this.H/2);

    ctx.beginPath();
    ctx.moveTo(MG+RD, MG);
    ctx.lineTo(this.W-MG-RD, 0+MG);
    ctx.quadraticCurveTo(this.W-MG, MG, this.W-MG, MG+RD); // right top
    ctx.lineTo(this.W-MG, this.H-MG-RD);
    ctx.quadraticCurveTo(this.W-MG, this.H-MG, this.W-MG-RD, this.H-MG); // right bottom
    ctx.lineTo(MG+RD, this.H-MG);
    ctx.quadraticCurveTo(MG, this.H-MG, MG, this.H-MG-RD); // left bottom
    ctx.lineTo(MG, MG+RD);
    ctx.quadraticCurveTo(MG, MG, MG+RD, MG); // left top
    ctx.closePath();
    ctx.fillStyle = FILL;
    ctx.fill();
    ctx.strokeStyle = STROKE;
    ctx.lineWidth = 2.0;
    ctx.stroke();

    var MG = this.W/10;
    var RD = this.W/12;
    ctx.beginPath();
    ctx.moveTo(MG, MG+RD);
    ctx.lineTo(MG, this.H-this.W/2.5);
    ctx.lineTo(this.W-MG, this.H-this.W/2.5);
    ctx.lineTo(this.W-MG, MG+RD);
    ctx.closePath();
    ctx.fillStyle = DISP_FILL;
    ctx.fill();
    ctx.strokeStyle = STROKE;
    ctx.lineWidth = 1.0;
    ctx.stroke();

    ctx.beginPath(); // home button
    ctx.arc(this.W/2, this.H-this.W/5, this.W/9, 0, 2*Math.PI, true);
    ctx.fillStyle = DISP_FILL;
    ctx.fill();
    ctx.strokeStyle = STROKE;
    ctx.lineWidth = 2.0;
    ctx.stroke();

    // ゆび
    if (this.finger > 0) {
            ctx.beginPath(); // home button
            ctx.arc(this.W*0.65, this.H*0.4, this.W*0.2, 0, 2*Math.PI, true);
            ctx.fillStyle = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
            ctx.fill();
            renderFinger(ctx, this.W*0.6, this.H*0.5-20, 40, 40, Math.PI*0.20, 1.0);
    }

    ctx.restore();
}

function renderFinger(ctx, x, y, W_OBJ, H_OBJ, angle, magnify) {
    ctx.strokeStyle ="hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
    ctx.fillStyle = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
    var cx = x+W_OBJ/2;
    var cy = y+H_OBJ/2;
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);

/*
    // 外枠（オプション）
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, H_OBJ);
    ctx.lineTo(W_OBJ, H_OBJ);
    ctx.lineTo(W_OBJ, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.stroke(); // strokeで輪郭。
*/
    ctx.beginPath();
    ctx.moveTo(W_OBJ*0.25, H_OBJ); // 手首内側
    ctx.quadraticCurveTo(W_OBJ*0.2, H_OBJ*0.85,  W_OBJ*0.175, H_OBJ*0.875);
    ctx.bezierCurveTo(W_OBJ*-0.2, H_OBJ*0.525,  W_OBJ*0.25, H_OBJ*0.20,  W_OBJ*0.2, H_OBJ*0.5); // おやゆび
    ctx.lineTo(W_OBJ*0.225, H_OBJ*0.45); // 親指の根元
    ctx.lineTo(W_OBJ*0.25, H_OBJ*0.375); // 親指の途中

    ctx.bezierCurveTo(W_OBJ*0.15, H_OBJ*-0.15,  W_OBJ*0.4, H_OBJ*-0.10,  W_OBJ*0.425, H_OBJ*0.3125);
    ctx.bezierCurveTo(W_OBJ*0.4, H_OBJ*0.3,  W_OBJ*0.55, H_OBJ*0.2,  W_OBJ*0.6, H_OBJ*0.35); // なかゆび
    ctx.bezierCurveTo(W_OBJ*0.6125, H_OBJ*0.3,  W_OBJ*0.7, H_OBJ*0.25,  W_OBJ*0.75, H_OBJ*0.375); // くすりゆび
    ctx.bezierCurveTo(W_OBJ*0.75, H_OBJ*0.3,  W_OBJ*0.925, H_OBJ*0.35,  W_OBJ*0.9, H_OBJ*0.5);// こゆび
    ctx.bezierCurveTo(W_OBJ*0.95, H_OBJ*0.7,  W_OBJ*0.8, H_OBJ*0.75,  W_OBJ*0.75, H_OBJ);

    ctx.lineTo(W_OBJ*0.75, H_OBJ); // 手首外側

    ctx.closePath();
    ctx.fill();
    ctx.stroke();

/*
    // ボディ中央マーカー
    ctx.beginPath(); // 新規パスを開始。ここから自機の描画処理。
    ctx.arc(W_OBJ/2, H_OBJ/2, 3, 0, Math.PI * 2, false); // 自機を描くパスを設定
    ctx.fill(); // 自機を描く
*/
}



// ============================================================================
// タイトルに載せるキーボードの絵。
// ============================================================================

function IlstKeys(depth, fromX, fromY, W, H) {
    this.depth = depth;
    this._x = fromX; // 定数。本来はconstを付けたいが、IE10以前で非対応なので諦める
    this._y = fromY; //
    this.W = W;
    this.H = H;
    this.angle = -Math.PI/2 ;
    this.magnify = 1.0;
    this.isDown32 = 0;
    this.isDown37 = 0;
    this.isDown38 = 0;
    this.isDown39 = 0;
    this.isDown40 = 0;
};


IlstKeys.prototype.setStatus = function (keys) {
    this.keys = keys;
    // left: 37, up: 38, right: 39, down: 40,
    // spacebar: 32, pageup: 33, pagedown: 34, end: 35, home: 36
    this.isDown32 = AsNumeric(this.keys.isDown(32));
    this.isDown37 = AsNumeric(this.keys.isDown(37));
    this.isDown38 = AsNumeric(this.keys.isDown(38));
    this.isDown39 = AsNumeric(this.keys.isDown(39));
    this.isDown40 = AsNumeric(this.keys.isDown(40));
}

AsNumeric = function(x) {
    var ans=0;
    if (x==true) {
        ans=1;
    }
    return ans;
}

// 毎フレーム、自身を描画させるメソッド。
IlstKeys.prototype.drawme = function (ctx) {

    this.font_family='font1';
    this.font_size = 40;
    this.size_unit = "px";
    ctx.lineWidth = 1;

    // 定数の初期化
    var FILL = "rgba(255, 255, 255, "+1+")"; // 塗りつぶし。
    var STROKE = "rgba(255,255, 255, 1)"; // 線色。
    var RD1 = 10;
    var RD2 = 5;

    ctx.save(); // 事前セーブ

    ctx.globalAlpha = 1;
    ctx.fillStyle = FILL;
    ctx.lineWidth = 2;
    ctx.strokeStyle = STROKE;

    // キーボード外枠
    ctx.beginPath();
    ctx.moveTo(this._x-0.07*this.W, this._y+this.H);
    ctx.lineTo(this._x+0.05*this.W+this.W-RD1, this._y+this.H);
    ctx.quadraticCurveTo( this._x+0.05*this.W+this.W, this._y+this.H,
                          this._x+0.05*this.W+this.W, this._y+this.H-RD1); // right top
    ctx.lineTo(this._x+0.05*this.W+this.W, this._y+0.1*this.H);
    ctx.stroke();

    var lw = 0.3*this.W;
    var lh = 0.3*this.H;
    var lr = 0.10*this.H;

    // 左キー
    var lx = this._x-0.025*this.W;
    var ly = this._y+ 0.55*this.H;
    ctx.beginPath();
    ctx.moveTo(lx, ly+lr);
    ctx.lineTo(lx, ly+lh-lr);
    ctx.quadraticCurveTo(lx, ly+lh, lx+lr, ly+lh);
    ctx.lineTo(lx+lw-lr, ly+lh);
    ctx.quadraticCurveTo(lx+lw, ly+lh, lx+lw, ly+lh-lr);
    ctx.lineTo(lx+lw, ly+lr);
    ctx.quadraticCurveTo(lx+lw, ly, lx+lw-lr, ly);
    ctx.lineTo(lx+lr, ly);
    ctx.quadraticCurveTo(lx, ly, lx, ly+lr);
    ctx.closePath();
    ctx.strokeStyle = "rgb("+255*(1-this.isDown37)+", "+255*(1-this.isDown37)+", "+255*(1-this.isDown37)+")";
    ctx.stroke();
    ctx.strokeStyle = STROKE;
    ctx.fillStyle = "rgba(255, 255, 255, "+this.isDown37+")"; // 一瞬だけ塗り色を変更
    ctx.fill();
    ctx.fillStyle = FILL;

    ctx.beginPath();
    ctx.moveTo(lx+0.4*lw, ly+0.5*lh);
    ctx.lineTo(lx+0.6*lw, ly+0.65*lh);
    ctx.lineTo(lx+0.6*lw, ly+0.35*lh);
    ctx.closePath();
    ctx.fillStyle = "rgb("+255*(1-this.isDown37)+", "+255*(1-this.isDown37)+", "+255*(1-this.isDown37)+")";
    ctx.fill();
    ctx.fillStyle = FILL;

    // 下キー
    var lx = this._x+0.325*this.W;
    var ly = this._y+ 0.55*this.H;
    ctx.beginPath();
    ctx.moveTo(lx, ly);
    ctx.lineTo(lx, ly+lh-lr);
    ctx.quadraticCurveTo(lx, ly+lh, lx+lr, ly+lh);
    ctx.lineTo(lx+lw-lr, ly+lh);
    ctx.quadraticCurveTo(lx+lw, ly+lh, lx+lw, ly+lh-lr);
    ctx.lineTo(lx+lw, ly);
    ctx.lineTo(lx, ly);
    ctx.closePath();
    ctx.strokeStyle = "rgb("+255*(1-this.isDown40)+", "+255*(1-this.isDown40)+", "+255*(1-this.isDown40)+")";
    ctx.stroke();
    ctx.strokeStyle = STROKE;
    ctx.fillStyle = "rgba(255, 255, 255, "+this.isDown40+")"; // 一瞬だけ塗り色を変更
    ctx.fill();
    ctx.fillStyle = FILL;

    ctx.beginPath();
    ctx.moveTo(lx+0.4*lw, ly+0.4*lh);
    ctx.lineTo(lx+0.5*lw, ly+0.75*lh);
    ctx.lineTo(lx+0.6*lw, ly+0.4*lh);
    ctx.closePath();
    ctx.fillStyle = "rgb("+255*(1-this.isDown40)+", "+255*(1-this.isDown40)+", "+255*(1-this.isDown40)+")";
    ctx.fill();
    ctx.fillStyle = FILL;

    // 右キー
    var lx = this._x+0.675*this.W;
    var ly = this._y+ 0.55*this.H;
    ctx.beginPath();
    ctx.moveTo(lx, ly+lr);
    ctx.lineTo(lx, ly+lh-lr);
    ctx.quadraticCurveTo(lx, ly+lh, lx+lr, ly+lh);
    ctx.lineTo(lx+lw-lr, ly+lh);
    ctx.quadraticCurveTo(lx+lw, ly+lh, lx+lw, ly+lh-lr);
    ctx.lineTo(lx+lw, ly+lr);
    ctx.quadraticCurveTo(lx+lw, ly, lx+lw-lr, ly);
    ctx.lineTo(lx+lr, ly);
    ctx.quadraticCurveTo(lx, ly, lx, ly+lr);
    ctx.closePath();
    ctx.strokeStyle = "rgb("+255*(1-this.isDown39)+", "+255*(1-this.isDown39)+", "+255*(1-this.isDown39)+")";
    ctx.stroke();
    ctx.strokeStyle = STROKE;
    ctx.fillStyle = "rgba(255, 255, 255, "+this.isDown39+")"; // 一瞬だけ塗り色を変更
    ctx.fill();
    ctx.fillStyle = FILL;

    ctx.beginPath();
    ctx.moveTo(lx+0.4*lw, ly+0.35*lh);
    ctx.lineTo(lx+0.4*lw, ly+0.65*lh);
    ctx.lineTo(lx+0.6*lw, ly+0.50*lh);
    ctx.closePath();
    ctx.fillStyle = "rgb("+255*(1-this.isDown39)+", "+255*(1-this.isDown39)+", "+255*(1-this.isDown39)+")";
    ctx.fill();
    ctx.fillStyle = FILL;

    // 上キー
    var lx = this._x+0.325*this.W;
    var ly = this._y+ 0.20*this.H;
    ctx.beginPath();
    ctx.moveTo(lx, ly+lr);
    ctx.lineTo(lx, ly+lh);
    ctx.lineTo(lx+lw, ly+lh);
    ctx.lineTo(lx+lw, ly+lr);
    ctx.quadraticCurveTo(lx+lw, ly, lx+lw-lr, ly);
    ctx.lineTo(lx+lr, ly);
    ctx.quadraticCurveTo(lx, ly, lx, ly+lr);
    ctx.closePath();
    ctx.strokeStyle = "rgb("+255*(1-this.isDown38)+", "+255*(1-this.isDown38)+", "+255*(1-this.isDown38)+")";
    ctx.stroke();
    ctx.strokeStyle = STROKE;
    ctx.fillStyle = "rgba(255, 255, 255, "+this.isDown38+")"; // 一瞬だけ塗り色を変更
    ctx.fill();
    ctx.fillStyle = FILL;

    ctx.beginPath();
    ctx.moveTo(lx+0.4*lw, ly+0.6*lh);
    ctx.lineTo(lx+0.6*lw, ly+0.6*lh);
    ctx.lineTo(lx+0.5*lw, ly+0.25*lh);
    ctx.closePath();
    ctx.fillStyle = "rgb("+255*(1-this.isDown38)+", "+255*(1-this.isDown38)+", "+255*(1-this.isDown38)+")";
    ctx.fill();
    ctx.fillStyle = FILL;

    // 事後ロード。これで拡大、オフセット、回転は全てリセットされ、後続の描画処理には影響しない。
    ctx.restore();
}




// ============================================================================
// 7-segments display の動作をエミュレートする。
// 元データのサイズは W 45px H 80px このままでは大抵の表示に大きすぎると思うので、適宜縮小する事。
// ============================================================================

// 2016/04/01 こいつの動作が極めて遅い理由は、おそらく配列に範囲外アクセスしているため。

// SSDクラスを自作する。こいつに数値データを食べさせて、文字列に変換し、内部の各数字をNanasegクラスオブジェクトとして格納し表示。
function SSD( depth_from, LEN_MAX, digit, where_x, where_y, h, col_R, col_G, col_B, data ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this.length_max = LEN_MAX;
    this.col_R = col_R;
    this.col_G = col_G;
    this.col_B = col_B;
    this._x = where_x;
    this._y = where_y
    this.h = h; // height だけ与えれば、アスペクト比から適切にwも計算できる。
    this.digit = digit;
    this.ary = new Array(this.length_max);
    for(var i = 0; i < this.length_max; i=(i+1)|0) {
        this.ary[i] = new Nanaseg((where_x + (h*0.6)*i), where_y, h);
    }
};


// 状態更新メソッド
// データは任意の実数、digitは何番目のセグメントの直後に小数点を入れるか。
// たとえばlength_max=8, digit=0なら.12345678、digit=4なら9999.1234 とか -999.1234とか。
// もしマイナスの値だったらどうすんの？ひとまず考えない。が、エラーメッセージは出ている


SSD.prototype.setStatus = function(num, where_x, where_y) {
    this._x = where_x;
    this._y = where_y;
    this.data = new Array(this.length_max);
    var kurai = Math.floor(Math.LOG10E * Math.log(Math.abs(num))); // 変数dataが何桁の数値であるか。
    // たとえば data=12345.678 ならば kurai = 5
    for(var i = this.length_max; i > 0; i=(i-1)|0) {
        var temp = num % Math.ceil(Math.pow(10, this.digit-i+1));
        this.data[i-1] = Math.floor(temp / Math.pow(10, this.digit-i));
    }
    for(var i = 0; i < this.length_max; i=(i+1)|0) {
        this.ary[i].setStatus((this._x + (this.h*0.6)*i), this._y, col_R, col_G, col_B, this.data[i]);
    }
};


// 描画メソッド
SSD.prototype.drawme = function(ctx) {
    for(var i = 0; i < this.length_max; i=(i+1)|0) {
        this.ary[i].drawme(ctx);
    }
};


// ============================================================================


// Nanasegクラスを定義する。こいつが一つ一つの数字を表示する素子である。
function Nanaseg(where_x, where_y, h) {
    this._x = where_x;
    this._y = where_y;
    this.angle = 0;
    this.magnify = h/80; // 当初のデザインは高さ80pxあるので、倍率で調整する。
    this.col_R = 0;
    this.col_G = 0;
    this.col_B = 0;
    this.data = 0;
};


// 描画準備。
Nanaseg.prototype.setStatus = function(where_x, where_y, col_R, col_G, col_B, data) {
    this._x = where_x;
    this._y = where_y;
    this.col_R = col_R;
    this.col_G = col_G;
    this.col_B = col_B;
    this.data = data;
};


// グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセット。
// rotateだけだと「左上の原点を中心に、反時計回りで」という直感に反した動きなので注意。
Nanaseg.prototype.offset = function(ctx, angle, magnify, where_x, where_y, W_CTX, H_CTX) {
    ctx.translate(where_x - W_CTX*magnify/2, where_y - H_CTX*magnify/2);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.translate(W_CTX/2, H_CTX/2);
    ctx.rotate( angle /180 * Math.PI);
    ctx.translate(-W_CTX/2, -H_CTX/2);
};


// 毎フレーム、自身を描画させるメソッド。

Nanaseg.prototype.drawme = function (ctx) {

    var i = this.data;

    // 消灯、点灯、半点灯
    var colo = ["rgba(" + this.col_R + ", " + this.col_G + ", " + this.col_B + ", 0.1)",
                "rgba(" + this.col_R + ", " + this.col_G + ", " + this.col_B + ", 1)",
                "rgba(" + this.col_R + ", " + this.col_G + ", " + this.col_B + ", 0.05)"];
    // 点灯パターンは任意の長さの配列として定義。
    // [0~9] は対応する数字を点灯させる。
    // [10]はドット、
    // [11]は全点灯、
    // [12--15]は消灯（未定義）
    var SEG_COLOR_A = [colo[1], colo[2], colo[1], colo[1],
                       colo[2], colo[1], colo[1], colo[1],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_B = [colo[1], colo[1], colo[1], colo[1],
                       colo[1], colo[2], colo[2], colo[1],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_C = [colo[1], colo[1], colo[2], colo[1],
                       colo[1], colo[1], colo[1], colo[1],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_D = [colo[1], colo[2], colo[1], colo[1],
                       colo[2], colo[1], colo[1], colo[2],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_E = [colo[1], colo[2], colo[1], colo[2],
                       colo[2], colo[2], colo[1], colo[2],
                       colo[1], colo[2], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_F = [colo[1], colo[2], colo[2], colo[2],
                       colo[1], colo[1], colo[1], colo[1],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_G = [colo[2], colo[2], colo[1], colo[1],
                       colo[1], colo[1], colo[1], colo[2],
                       colo[1], colo[1], colo[2], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var SEG_COLOR_DP =[colo[2], colo[2], colo[2], colo[2],
                       colo[2], colo[2], colo[2], colo[2],
                       colo[2], colo[2], colo[1], colo[1],
                       colo[0], colo[0], colo[0], colo[0]];

    var W_CTX = 45; // 1つの数字の描画領域の幅。
    var H_CTX = 80; // 1つの数字の描画領域の高さ。

    ctx.save();

    // グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセットして回転を掛ける。
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, W_CTX, H_CTX);

    // 外枠（オプション）
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, H_CTX);
    ctx.lineTo(W_CTX, H_CTX);
    ctx.lineTo(W_CTX, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.strokeStyle = "rgba(5, 5, 5, 1)";
//    ctx.stroke(); // strokeで輪郭。

    // ここから、キャラクター毎の形状に依存した描画処理。

    // segments/G
    ctx.beginPath();
    ctx.moveTo(11.0, 44.5);
    ctx.bezierCurveTo(8.5, 44.5, 6.0, 41.9, 4.3, 40.0);
    ctx.bezierCurveTo(4.2, 40.0, 4.1, 39.9, 4.1, 39.9);
    ctx.bezierCurveTo(4.2, 39.8, 4.3, 39.7, 4.3, 39.6);
    ctx.bezierCurveTo(6.4, 37.5, 8.8, 35.1, 11.2, 35.1);
    ctx.lineTo(25.7, 35.2);
    ctx.bezierCurveTo(27.8, 35.2, 30.0, 37.7, 31.8, 39.7);
    ctx.bezierCurveTo(31.9, 39.9, 32.0, 40.0, 32.2, 40.1);
    ctx.bezierCurveTo(29.7, 42.4, 26.7, 44.6, 25.1, 44.6);
    ctx.lineTo(11.0, 44.5);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_G[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/F
    ctx.beginPath();
    ctx.moveTo(2.9, 38.3);
    ctx.bezierCurveTo(2.4, 37.2, 2.1, 35.6, 2.1, 34.0);
    ctx.bezierCurveTo(2.1, 31.4, 3.6, 7.7, 3.7, 6.4);
    ctx.bezierCurveTo(3.8, 5.9, 4.1, 5.1, 4.6, 4.2);
    ctx.lineTo(11.6, 10.2);
    ctx.lineTo(10.3, 33.3);
    ctx.bezierCurveTo(7.5, 33.7, 5.1, 36.1, 3.0, 38.2);
    ctx.bezierCurveTo(3.0, 38.2, 2.9, 38.3, 2.9, 38.3);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_F[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/E
    ctx.beginPath();
    ctx.moveTo(0.0, 72.7);
    ctx.bezierCurveTo(0.1, 70.9, 0.4, 65.7, 0.7, 60.3);
    ctx.lineTo(0.7, 60.2);
    ctx.bezierCurveTo(1.1, 53.6, 1.5, 46.7, 1.5, 45.4);
    ctx.bezierCurveTo(1.5, 43.5, 2.2, 42.1, 2.8, 41.3);
    ctx.bezierCurveTo(2.8, 41.3, 2.8, 41.3, 2.8, 41.4);
    ctx.bezierCurveTo(4.5, 43.2, 6.8, 45.6, 9.5, 46.3);
    ctx.lineTo(8.3, 69.5);
    ctx.lineTo(0.0, 72.7);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_E[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/D
    ctx.beginPath();
    ctx.moveTo(10.1, 80.0);
    ctx.bezierCurveTo(2.3, 79.9, 0.5, 76.3, 0.1, 74.8);
    ctx.lineTo(9.4, 71.2);
    ctx.lineTo(22.9, 71.2);
    ctx.lineTo(25.4, 80.0);
    ctx.lineTo(10.1, 80.0);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_D[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/C
    ctx.beginPath();
    ctx.moveTo(24.6, 70.1);
    ctx.lineTo(25.9, 46.4);
    ctx.bezierCurveTo(28.4, 45.9, 31.7, 43.2, 33.3, 41.7);
    ctx.bezierCurveTo(33.8, 42.8, 34.2, 44.4, 34.2, 46.0);
    ctx.bezierCurveTo(34.2, 48.8, 32.7, 72.4, 32.5, 73.6);
    ctx.bezierCurveTo(32.1, 76.5, 28.9, 78.6, 27.3, 79.4);
    ctx.lineTo(24.6, 70.1);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_C[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/B
    ctx.beginPath();
    ctx.moveTo(33.2, 38.4);
    ctx.bezierCurveTo(31.4, 36.4, 29.3, 33.9, 26.7, 33.4);
    ctx.lineTo(27.9, 10.1);
    ctx.lineTo(32.9, 1.9);
    ctx.bezierCurveTo(34.6, 3.3, 36.2, 5.1, 36.2, 6.7);
    ctx.bezierCurveTo(36.2, 7.7, 35.9, 13.5, 35.6, 19.6);
    ctx.lineTo(35.6, 19.7);
    ctx.bezierCurveTo(35.2, 26.4, 34.8, 33.3, 34.8, 34.6);
    ctx.bezierCurveTo(34.8, 36.5, 34.1, 37.9, 33.5, 38.7);
    ctx.bezierCurveTo(33.4, 38.6, 33.3, 38.5, 33.2, 38.4);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_B[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/A
    ctx.beginPath();
    ctx.moveTo(12.9, 8.8);
    ctx.lineTo(5.9, 2.7);
    ctx.bezierCurveTo(7.3, 1.3, 9.2, 0.0, 10.7, 0.0);
    ctx.lineTo(26.1, 0.0);
    ctx.bezierCurveTo(28.3, 0.1, 30.1, 0.6, 31.2, 0.9);
    ctx.lineTo(26.4, 8.8);
    ctx.lineTo(12.9, 8.8);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_A[i]; // 塗りつぶしスタイル
    ctx.fill();

    // segments/DP
    ctx.beginPath();
    ctx.moveTo(40.5, 80.0);
    ctx.bezierCurveTo(38.0, 80.0, 36.0, 78.0, 36.0, 75.5);
    ctx.bezierCurveTo(36.0, 73.0, 38.0, 71.0, 40.5, 71.0);
    ctx.bezierCurveTo(41.7, 71.0, 42.8, 71.5, 43.7, 72.3);
    ctx.bezierCurveTo(44.5, 73.2, 45.0, 74.3, 45.0, 75.5);
    ctx.bezierCurveTo(45.0, 78.0, 43.0, 80.0, 40.5, 80.0);
    ctx.lineTo(40.5, 80.0);
    ctx.closePath();
    ctx.fillStyle = SEG_COLOR_DP[i]; // 塗りつぶしスタイル
    ctx.fill();

    // ここまで、キャラクター毎の形状に依存した描画処理。

    // ボディ中央マーカー
    ctx.beginPath(); // 新規パスを開始。ここから自機の描画処理。
    ctx.arc(W_CTX/2, H_CTX/2, 3, 0, Math.PI*2, false); // 自機を描くパスを設定
    ctx.fillStyle = "rgb(0, 0, 0)"; // 自機の色を設定する
//    ctx.fill(); // 自機を描く

    ctx.restore();
}



// ============================================================================
// 2016/05/22 AnalogueMeter クラスを自作する。
// まあ簡単に言うと、予め指定しておいた長さndataの配列の形で、一連の数値を入力し、
// 扇形のレンジの上を針が動いて数値を表示する。配列の最初の要素を最新フレームの値とし、毎フレーム数値を右にずらして更新。
// 針の角度を毎フレーム最新の値だけで更新させると振れが大きすぎて意味不明になるので、直前の一定フレームを
// 何らかのカーネルで重み付けした平均値として、実際に使う表示値を決める。
// レンジは予め初期化時に指定しておく。実際の値が初期レンジを超えたら、表示は右端に固定すべきだが、あまり深く考えていない。
// ============================================================================


function AnalogueMeter( depth_from, ndata, d_default, lim, where_x, where_y, W, H,
                        font_family, font_size, size_unit) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this.d_default = d_default; // default は、このメーターの値が示すと期待されるデフォルト。
    this.lim = lim;
    this._x = where_x;
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.angle = 0;
    this.magnify = 1;
    this.ndata = ndata;
    this.rawdata = new Array(ndata);
    for (var i = 0; i < ndata; i=(i+1)|0) {
        this.rawdata[i] = 0; // 配列要素を全てゼロに
    }

    this.kernel = new Array(ndata); // 重み付け処理用の加重
    for (var i = 0; i < ndata; i=(i+1)|0) {
        this.kernel[i] = 1; // たとえばカーネルを全部1で初期化＝新しいデータも古いデータも重みは同じとした場合（一様）
//        this.kernel[i] = 1 - i/ndata; // 三角形。古い情報の価値は直線的に下がる
    }
    this.kernel_sum = 0;
    for (var i = 0; i < ndata; i=(i+1)|0) {
        this.kernel_sum = this.kernel_sum + this.kernel[i];
    }
    this.weighed = this.rawdata.slice();
    this.data = d_default;
    this.font_family=font_family; this.font_size = font_size; this.size_unit = size_unit;
    this.pss = 0;
};


// 状態更新メソッド。 dataは配列ではなく、最新の値を1つだけ与えるスカラー。
// 欠測時にはデフォルト値で自動的に補完される。これがいやなら、代入時に前の値を使わせる等の具体的な指定を行うこと。

AnalogueMeter.prototype.setStatus = function(data, pss) {
    for (var i = ndata-1; i>=0; i=(i-1)|0) {
        this.rawdata[i+1] = this.rawdata[i]; // 配列要素を1つずつ右にずらして代入。最初の要素が古いままなので、次いで↓
    }
    if (isFinite(data) == true) {
        this.rawdata[0] = data;
    } else {
        this.rawdata[0] = this.d_default;
    }
    // 次いで表示用のデータを作成。
    for (var i = 0; i < ndata; i=(i+1)|0) {
        this.weighed[i] = this.rawdata[i]*this.kernel[i] / this.kernel_sum;
    }
    this.data = 0;
    for (var i = 0; i < ndata; i=(i+1)|0) {
        this.data = this.data + this.weighed[i];
    }
    this.pss = pss; // 主針がゲーム全体の実効FPS、こちらはステージ関数の実行にかかった本来の時間。
};



// 描画メソッド
AnalogueMeter.prototype.drawme = function(ctx) {

    ctx.save(); // 事前セーブ

    // グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセットして回転を掛ける。
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H);
    ctx.textAlign = "center";
    var METER_COL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var FONT_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var FONT = this.font_size + this.size_unit + " " + this.font_family;
    ctx.font = FONT;

    // レンジを途中で変える必要が無ければ、このパラメータ群は、最初に1回定義すればいい。
    var e = this.lim[0];
    var g = this.lim[1]-this.lim[0];
    var scaleMain = [e+g*0.1, e+g*0.3, e+g*0.5, e+g*0.7, e+g*0.9];
    var scaleSub1 = [e, e+g*0.2, e+g*0.4, e+g*0.6, e+g*0.8, e+g*1.0];
    var scaleMain_len = scaleMain.length;
    var scaleSub1_len = scaleSub1.length;
//    var start_rad = (1/12 + 1) * Math.PI; // start_rad==0の時に180度（左から開始）になるようにしている。そこから
//    var range_rad = 10/12 * Math.PI; // 時計回りにrange_radだけ開いた扇形になる。
    // 上が標準的な設定で、下は横倒しになっているバージョン
    var start_rad = (-1/2 + 1) * Math.PI; // start_rad==0の時に180度（左から開始）になるようにしている。そこから
    var range_rad = 15/12 * Math.PI; // 時計回りにrange_radだけ開いた扇形になる。
    var india = 50;
    var outdia = 57;
    var outdia_sub = 55;
    var harimoto = 10;
    var harisaki = 60;
    var harithick = 0.2;

    // arc
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, india, start_rad, start_rad+range_rad, false);
    ctx.lineWidth = 1;
    ctx.strokeStyle = METER_COL;
    ctx.stroke();

    // Scale Main
    for(var i = 0; i < scaleMain_len; i=(i+1)|0) {
        var angle = start_rad + range_rad*scaleMain[i]/g;
        var tx = Math.cos(start_rad + range_rad*scaleMain[i]/g);
        var ty = Math.sin(start_rad + range_rad*scaleMain[i]/g);
        ctx.beginPath();
        ctx.moveTo( this.W/2 + india*tx, this.H/2 + india*ty );
        ctx.lineTo( this.W/2 + outdia*tx, this.H/2 + outdia*ty );
        ctx.lineWidth = 2.5;
        ctx.strokeStyle = METER_COL;
        ctx.stroke();

        // labels for the main scale
        ctx.translate(this.W/2, this.H/2);
        ctx.rotate(Math.PI/2 + angle);
        ctx.fillStyle = FONT_FILL;
        ctx.fillText(text=scaleMain[i], x=0, y=-(outdia+5), maxWidth=80);
        ctx.rotate(-Math.PI/2 - angle);
        ctx.translate(-this.W/2, -this.H/2);
    }

    // Scale Sub1
    for(var i = 0; i < scaleSub1_len; i=(i+1)|0) {
        var angle = start_rad + range_rad*scaleSub1[i]/g;
        var tx = Math.cos(start_rad + range_rad*scaleSub1[i]/g);
        var ty = Math.sin(start_rad + range_rad*scaleSub1[i]/g);
        ctx.beginPath();
        ctx.moveTo( this.W/2 + india*tx, this.H/2 + india*ty );
        ctx.lineTo( this.W/2 + outdia_sub*tx, this.H/2 + outdia_sub*ty );
        ctx.lineWidth = 1;
        ctx.strokeStyle = METER_COL;
        ctx.stroke();
    }

    // ボディ中央マーカー
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, this.d_default, 0, Math.PI * 2, false);
    ctx.fillStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
    ctx.fill();

    // ボディ中央マーカー 可動部
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, this.data, 0, Math.PI * 1, false);
    ctx.fillStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)";
    ctx.fill();

    // default value indicator
    var angle = start_rad + range_rad*this.d_default/g;
    ctx.beginPath();
    ctx.moveTo( this.W/2 + (india-1)*Math.cos(angle), this.H/2 + (india-1)*Math.sin(angle) );
    ctx.lineTo( this.W/2 + (india-7)*Math.cos(angle-0.05), this.H/2 + (india-7)*Math.sin(angle-0.05) );
    ctx.lineTo( this.W/2 + (india-7)*Math.cos(angle+0.05), this.H/2 + (india-7)*Math.sin(angle+0.05) );
    ctx.closePath();
    ctx.fillStyle = METER_COL;
    ctx.fill();
    ctx.lineWidth = 0.1;
    ctx.strokeStyle = METER_COL;
    ctx.stroke();

    // PSS indicator
    var angle = start_rad + range_rad*this.pss/g;
    ctx.beginPath();
    ctx.moveTo( this.W/2 + (india-1)*Math.cos(angle), this.H/2 + (india-1)*Math.sin(angle) );
    ctx.lineTo( this.W/2 + (india-7)*Math.cos(angle-0.05), this.H/2 + (india-7)*Math.sin(angle-0.05) );
    ctx.lineTo( this.W/2 + (india-7)*Math.cos(angle+0.05), this.H/2 + (india-7)*Math.sin(angle+0.05) );
    ctx.closePath();
    ctx.fillStyle = METER_COL;
    ctx.fill();
    ctx.lineWidth = 0.1;
    ctx.strokeStyle = METER_COL;
    ctx.stroke();


    // Indicator
    var tx = Math.cos(start_rad + range_rad*this.data/g);
    var ty = Math.sin(start_rad + range_rad*this.data/g);
    ctx.beginPath();
    ctx.moveTo( this.W/2 - tx*harimoto, this.H/2 - ty*harimoto);
    ctx.lineTo( this.W/2 + tx*harisaki, this.H/2 + ty*harisaki);
    ctx.lineWidth = 1.5;
    ctx.strokeStyle = METER_COL;
//    ctx.stroke();

    // default value indicator
    angle = start_rad + range_rad*this.data/g;
    ctx.beginPath();
    ctx.moveTo( this.W/2 + harisaki*Math.cos(angle), this.H/2 + harisaki*Math.sin(angle) );
    ctx.lineTo( this.W/2 + (-harimoto)*Math.cos(angle-harithick), this.H/2 + (-harimoto)*Math.sin(angle-harithick) );
    ctx.lineTo( this.W/2 + (-harimoto)*Math.cos(angle+harithick), this.H/2 + (-harimoto)*Math.sin(angle+harithick) );
    ctx.closePath();
    ctx.lineWidth = 0.5;
    ctx.strokeStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.semiCol[2]+"%, 1)"; // 塗りつぶし。
    ctx.stroke();
    ctx.fillStyle = METER_COL;
    ctx.fill();


    // title
    var angle = 0;
    ctx.translate(this.W/2, this.H/2);
    ctx.rotate(angle);
    ctx.fillStyle = FONT_FILL;
    ctx.fillText(text="FPS", x=0, y=-30, maxWidth=80);
    ctx.rotate(-angle);
    ctx.translate(-this.W/2, -this.H/2);

    ctx.restore();
};


// グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセット。
// rotateだけだと「左上の原点を中心に、反時計回りで」という直感に反した動きなので注意。
AnalogueMeter.prototype.offset = function(ctx, angle, magnify, where_x, where_y, W_CTX, H_CTX) {
    ctx.translate(where_x - W_CTX*magnify/2, where_y - H_CTX*magnify/2);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.translate(W_CTX/2, H_CTX/2);
    ctx.rotate( angle /180 * Math.PI);
    ctx.translate(-W_CTX/2, -H_CTX/2);
};


// ============================================================================
// Config クラスを自作する。本来のゲームキャンバスとは別に、詳細設定用のフロートウィンドウを出す。
// 普段は小さな形で表示しておき、クリックすると横から「にゅっ」と顔を出す。

// 正確にいうとこのクラスはconfig画面の出し入れに使うクリック反応領域を与える。実際に出し入れされるのはcanvas要素である。
// ============================================================================


function Config( depth_from, from, where_x, where_y, W, H, Whide, Hhide, radius, isopen,
                        font_family, font_size, size_unit, text_array ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this.from = from; // "top" "right" "bottom" "left" のいずれかで指定。現在はleftとrightしか実装していないので注意。
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.Whide = Whide;
    this.Hhide = Hhide;
    this.isopen = isopen; // 表示状態 1 か非表示状態 0 か
    this.onMouse = false; // ボタンにマウスが被っていればtrue
    this.power = 0;
    this.angle = 0;
    this.magnify = 1;
    this.radius = radius;
    this.font_family=font_family; this.font_size = font_size; this.size_unit = size_unit;
    this.text = text_array;
};


// 状態更新メソッド。

Config.prototype.setStatus = function(power) {
    this.power = power;
}

// 自身を描画。
// このクラスは特殊で、not open の状態ではオブジェクト全体がボタンとして反応する。
// 一方open時は、内部に他の設定ボタンやスイッチを備えるので、オブジェクトの地を反応させてはいけない。
// このときは「閉じる」ボタンだけを対応させる。

Config.prototype.drawme = function (ctx) {

    ctx.save(); // 事前セーブ

    if (this.power > 0) {
        var show = 1;
        var wi = this.W;
        var hi = this.H;
    } else {
        var show = 0;
        var wi = this.Whide;
        var hi = this.Hhide;
    }

    ctx.lineWidth = 1.0;

    if(this.onMouse == false) {
        ctx.strokeStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, "+show+")";
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, "+show+")";
    } else {
        ctx.strokeStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0)";
        ctx.fillStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0)"; // 塗りつぶしスタイル
    }

    // グローバル座標上のボタン描画開始位置を指定位置に移動。
    this.offset(ctx, this.angle, this.magnify, this._x, this._y);
    if (this.from =="right") {
        this.radiusUL = this.radius;
        this.radiusBL = this.radius*(1-this.power);
        this.radiusBR = 0;
        this.radiusUR = 0;
    }
    if (this.from =="left") {
        ctx.translate((this.W-this.Whide)*(1-this.power), 0);
        this.radiusUL = 0;
        this.radiusBL = 0;
        this.radiusBR = this.radius*(1-this.power);
        this.radiusUR = this.radius;
    }

    // 二次ベジエ曲線による角丸四角形。quadraticCurveToの最初の2つの引数が制御点で、四角形の頂点にするといい。
    // 上述の通りパス定義を繰り返す。1回目はstrokeやfillで本体の描画を行わなず、内外判定だけに使う
    ctx.beginPath();
    ctx.moveTo(0, this.radiusUL);
    ctx.lineTo(0, hi-this.radiusBL);
    ctx.quadraticCurveTo(0, hi, this.radiusBL, hi); // ひだりした
    ctx.lineTo(wi-this.radiusBR, hi);
    ctx.quadraticCurveTo(wi, hi, wi, hi-this.radiusBR); // みぎした
    ctx.lineTo(wi, this.radiusUR);
    ctx.quadraticCurveTo(wi, 0, wi-this.radiusUR, 0); // みぎうえ
    ctx.lineTo(this.radiusUL, 0);
    ctx.quadraticCurveTo(0, 0, 0, this.radiusUL); // ひだりうえ
    ctx.closePath();

    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。

    // 上記と同じポリゴンだが、もう1回beginPath()する
    ctx.beginPath();
    ctx.moveTo(0, this.radiusUL);
    ctx.lineTo(0, hi-this.radiusBL);
    ctx.quadraticCurveTo(0, hi, this.radiusBL, hi); // ひだりした
    ctx.lineTo(wi-this.radiusBR, hi);
    ctx.quadraticCurveTo(wi, hi, wi, hi-this.radiusBR); // みぎした
    ctx.lineTo(wi, this.radiusUR);
    ctx.quadraticCurveTo(wi, 0, wi-this.radiusUR, 0); // みぎうえ
    ctx.lineTo(this.radiusUL, 0);
    ctx.quadraticCurveTo(0, 0, 0, this.radiusUL); // ひだりうえ
    ctx.closePath();

    ctx.fill();
    ctx.stroke(); // strokeで輪郭。

    ctx.lineWidth = 6;
    ctx.lineJoin = 'round';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillStyle = '#fff';
    if (this.onMouse > 0) {
        ctx.fillText( this.text[this.power], (wi+(this.radiusUL-this.radiusUR)*show)/2, hi/2, wi);
    }

    ctx.restore();
};


// グローバル座標上のキャラクター位置を指定位置に移動。
// ちょっと特殊な性質があって、
// this.from=="left"のときは起点は左上に。
// this.from=="right"のときは起点は右上に。
// this.from=="top"のときは起点は中央上に。
// this.from=="bottom"のときは起点は中央下に。

Config.prototype.offset = function(ctx, rad, magnify, where_x, where_y) {
    ctx.translate(where_x, where_y);
    ctx.scale(magnify, magnify); // サイズ変更
    if (rad != 0) {
        ctx.rotate( rad ); // ラジアンである事に注意。
    }
};



// ============================================================================
// Config 用のアイコン
// 問題はどうやってコンフィグ画面の展開・収縮に併せて描画位置を変えさせるか。ctx3自体に展開機能を持たせるのが楽。
// 普通に考えたらcanvas要素であるconfigの表示幅を、DOMから操作する事になる。
// wx3の変更を、本体に波及させるには？実はすごく簡単。取得したgetElementIDを使い、width属性に値を代入するだけ。


function IconConfig( depth_from, where_x, where_y, W, H ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.onMouse = false; // ボタンにマウスが被っていればtrue。あたり判定が描画関数の中で定義されることに注意。
    this.power = 0;
    this.angle = -Math.PI/2;
    this.magnify = 1;
    this.radius = 10;
};


// 状態更新メソッド。
IconConfig.prototype.setStatus = function(power) {
    this.power = power;
}


// 自身を描画。
IconConfig.prototype.drawme = function (ctx) {


    if(this.onMouse == false) {
        ctx.strokeStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0)";
        ctx.fillStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0.91)";
    } else {
        ctx.strokeStyle = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
    }

    // zは歯の枚数。
    var z = 8;
    var ad = Math.min(this.W/2, this.H/2); // addendumは歯車の山の部分の半径
    var tl = 0.057; // top land
    var rt = 0.75*ad; // rootは歯車の歯の根本部分の半径
    var ax = 0.35*ad; // axisは歯車の回転軸部分の穴の半径

    ctx.save();

    this.offset(ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H);
    ctx.translate(ad, ad); // 最初に中心を半径分だけ移動し、位置合わせ

    ctx.beginPath();
    ctx.moveTo(ad*Math.cos(2*Math.PI*(0/z)-tl), ad*Math.sin(2*Math.PI*(0/z)-tl));
    for (var i = 0; i < z; i=(i+1)|0) {
        ctx.lineTo(ad*Math.cos(2*Math.PI*(i/z)-tl), ad*Math.sin(2*Math.PI*(i/z)-tl));
        ctx.lineTo(rt*Math.cos(2*Math.PI*(i/z)+tl), rt*Math.sin(2*Math.PI*(i/z)+tl));
        ctx.lineTo(rt*Math.cos(2*Math.PI*((i+0.5)/z)-tl), rt*Math.sin(2*Math.PI*((i+0.5)/z)-tl));
        ctx.lineTo(ad*Math.cos(2*Math.PI*((i+0.5)/z)+tl), ad*Math.sin(2*Math.PI*((i+0.5)/z)+tl));
    }
    ctx.closePath();

    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。

    // 上記と同じポリゴンだが、もう1回beginPath()する
    ctx.beginPath();
    ctx.moveTo(ad*Math.cos(2*Math.PI*(0/z)-tl), ad*Math.sin(2*Math.PI*(0/z)-tl));
    for (var i = 0; i < z; i=(i+1)|0) {
        ctx.lineTo(ad*Math.cos(2*Math.PI*(i/z)-tl), ad*Math.sin(2*Math.PI*(i/z)-tl));
        ctx.lineTo(rt*Math.cos(2*Math.PI*(i/z)+tl), rt*Math.sin(2*Math.PI*(i/z)+tl));
        ctx.lineTo(rt*Math.cos(2*Math.PI*((i+0.5)/z)-tl), rt*Math.sin(2*Math.PI*((i+0.5)/z)-tl));
        ctx.lineTo(ad*Math.cos(2*Math.PI*((i+0.5)/z)+tl), ad*Math.sin(2*Math.PI*((i+0.5)/z)+tl));
    }
    ctx.closePath();
    ctx.arc(0, 0, ax, Math.PI*2, 0, true);

    ctx.lineWidth = 1.0;
    ctx.stroke();
    ctx.fill();

    ctx.restore();
};


// グローバル座標上のキャラクター位置を指定位置に移動。
IconConfig.prototype.offset = function(ctx, angle, magnify, where_x, where_y, W_CTX, H_CTX) {
    ctx.translate(where_x , where_y );
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.translate(W_CTX/2, H_CTX/2);
    ctx.rotate( angle /180 * Math.PI);
    ctx.translate(-W_CTX/2, -H_CTX/2);
};


// ============================================================================
// リンク設定画面の背景。
// ============================================================================

function LinkBase( depth_from, angle, magnify, where_x, where_y, W, H, connected ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.onMouse = false; // ボタンにマウスが被っていればtrue。あたり判定が描画関数の中で定義されることに注意。
    this.connected = connected; // リンクがつながっているか否かの状態変数
    this.angle = angle;
    this.magnify = magnify;
};

// 状態更新メソッド。
LinkBase.prototype.setStatus = function(connected) {
    this.connected = connected;
}

// 自身を描画。実はポリゴン定義が面倒だったので寸法が設定値より大きくなってしまうが、まあ面倒なので放置。
LinkBase.prototype.drawme = function (ctx, transparency) {
    var RD = this.H*0.5;

    ctx.save(); // 事前セーブ
    ctx.lineCap = "butt";
    if(this.onMouse == false) {
        ctx.fillStyle = "hsla("+ctx.baseCol[0]+","+ctx.baseCol[1]+"%,"+ctx.baseCol[2]+"%,"+transparency+")";
    } else {
        ctx.fillStyle = "hsla("+ctx.darkCol[0]+","+ctx.darkCol[1]+"%,"+ctx.darkCol[2]+"%, "+transparency+")";
    }
    
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H, this._x+this.W/2, this._y+this.H/2);

    ctx.beginPath();
    ctx.arc(-this.W/2+RD, 0, RD, Math.PI*0.5, Math.PI*1.5, false);
    ctx.arc(this.W/2-RD, 0, RD, Math.PI*1.5, Math.PI*2.5, false);
    ctx.closePath();

    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。

    ctx.beginPath();
    ctx.arc(-this.W/2+RD, 0, RD, Math.PI*0.5, Math.PI*1.5, false);
    ctx.arc(this.W/2-RD, 0, RD, Math.PI*1.5, Math.PI*2.5, false);
    ctx.closePath();
    ctx.fill();

    ctx.restore();
}


// 無回転時の基点を左上とし、任意の場所を中心に回転できる
LinkBase.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};


// ============================================================================
// リンクチェーンのアイコン
// ============================================================================


function IconChain( depth_from, angle, magnify, where_x, where_y, W, H, connected ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.onMouse = false; // ボタンにマウスが被っていればtrue。あたり判定が描画関数の中で定義されることに注意。
    this.connected = connected; // リンクがつながっているか否かの状態変数
    this.angle = angle;
    this.magnify = magnify;
};

// 状態更新メソッド。
IconChain.prototype.setStatus = function(connected) {
    this.connected = connected;
}

// 自身を描画。実はポリゴン定義が面倒だったので寸法が設定値より大きくなってしまうが、まあ面倒なので放置。
IconChain.prototype.drawme = function (ctx) {

    var ad = Math.min(this.W/4, this.H/4); // adは鎖の円弧部分の半径
    var tl = 0.15; // 鎖の開口部の角度の半分
    var tp = 0.3; // 鎖の一番高い所の角度
    var rt = 1.85*ad; //
    var ax = 0.35*ad; //

    ctx.save(); // 事前セーブ

    ctx.lineCap = "butt";

    this.offset(ctx, this.angle, this.magnify, this._x-this.W/2, this._y-this.H/2, this.W, this.H, this._x+this.W/2, this._y+this.H/2);

    // 外枠（当たり判定：図形が細いので、パスに判定をつけると操作性が落ちる）
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, ad*3.5, 0, -Math.PI*2, true);
    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。

    if(this.onMouse == false) {
        ctx.strokeStyle = "hsla("+ctx.liteCol[0]+","+ctx.liteCol[1]+"%,"+ctx.liteCol[2]+"%,"+(0.4+this.connected*0.6)+")";
    } else {
        ctx.strokeStyle = "hsla("+ctx.darkCol[0]+","+ctx.darkCol[1]+"%,"+ctx.darkCol[2]+"%, "+(0.6+this.connected*0.4)+")";
    }
    ctx.lineWidth = 0.4*ad;

    ctx.beginPath();
    ctx.arc(this.W/2-rt+ax, this.H/2, ad, -Math.PI*(tl), -Math.PI*(tp), true);
    ctx.arc(this.W/2-rt-ax, this.H/2, ad, -Math.PI*(1-tp), -Math.PI*(1+tp), true);
    ctx.arc(this.W/2-rt+ax, this.H/2, ad, -Math.PI*(2-tp), -Math.PI*(2-tl), true);
    ctx.stroke(); // strokeで輪郭。

    ctx.beginPath();
    ctx.arc(this.W/2+rt-ax, this.H/2, ad, -Math.PI*(1-tl), -Math.PI*(1-tp), false);
    ctx.arc(this.W/2+rt+ax, this.H/2, ad, -Math.PI*(tp), -Math.PI*(-tp), false);
    ctx.arc(this.W/2+rt-ax, this.H/2, ad, -Math.PI*(-1+tp), -Math.PI*(-1+tl), false);
    ctx.stroke(); // strokeで輪郭。

    if (this.connected > 0) {
        ctx.lineCap = "round";
        ctx.beginPath();
        ctx.moveTo(this.W/2-rt, this.H/2);
        ctx.lineTo(this.W/2+rt, this.H/2);
        ctx.stroke(); // strokeで輪郭。
        ctx.lineCap = "butt";
    } else {
        ctx.lineCap = "round";

        ctx.beginPath();
        ctx.moveTo(this.W/2-rt, this.H/2);
        ctx.lineTo(this.W/2-rt/2, this.H/2);
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath();
        ctx.moveTo(this.W/2+rt/2, this.H/2);
        ctx.lineTo(this.W/2+rt, this.H/2);
        ctx.stroke(); // strokeで輪郭。

        ctx.lineWidth = 0.35*ad;

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*1.75*Math.cos(-Math.PI*0.5), this.H/2 + ad*1.75*Math.sin(-Math.PI*0.5) );
        ctx.lineTo( this.W/2 + ad*2.5*Math.cos(-Math.PI*0.5), this.H/2 + ad*2.5*Math.sin(-Math.PI*0.5) );
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*2*Math.cos(-Math.PI*0.3), this.H/2 + ad*2*Math.sin(-Math.PI*0.3) );
        ctx.lineTo( this.W/2 + ad*2.75*Math.cos(-Math.PI*0.3), this.H/2 + ad*2.75*Math.sin(-Math.PI*0.3) );
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*2*Math.cos(-Math.PI*0.7), this.H/2 + ad*2*Math.sin(-Math.PI*0.7) );
        ctx.lineTo( this.W/2 + ad*2.75*Math.cos(-Math.PI*0.7), this.H/2 + ad*2.75*Math.sin(-Math.PI*0.7) );
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*1.75*Math.cos(-Math.PI*1.5), this.H/2 + ad*1.75*Math.sin(-Math.PI*1.5) );
        ctx.lineTo( this.W/2 + ad*2.5*Math.cos(-Math.PI*1.5), this.H/2 + ad*2.5*Math.sin(-Math.PI*1.5) );
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*2*Math.cos(-Math.PI*1.3), this.H/2 + ad*2*Math.sin(-Math.PI*1.3) );
        ctx.lineTo( this.W/2 + ad*2.75*Math.cos(-Math.PI*1.3), this.H/2 + ad*2.75*Math.sin(-Math.PI*1.3) );
        ctx.stroke(); // strokeで輪郭。

        ctx.beginPath(); //
        ctx.moveTo( this.W/2 + ad*2*Math.cos(-Math.PI*1.7), this.H/2 + ad*2*Math.sin(-Math.PI*1.7) );
        ctx.lineTo( this.W/2 + ad*2.75*Math.cos(-Math.PI*1.7), this.H/2 + ad*2.75*Math.sin(-Math.PI*1.7) );
        ctx.stroke(); // strokeで輪郭。

        ctx.lineCap = "butt";
    }
    // ここまで、キャラクター毎の形状に依存した描画処理。

    ctx.restore();
};



// 無回転時の基点を左上とし、任意の場所を中心に回転できる
IconChain.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};


// ============================================================================
// リンク設定画面用の葉っぱのアイコン
// ============================================================================


function IconLeaf( depth_from, angle, magnify, where_x, where_y, W, H) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.onMouse = false; // ボタンにマウスが被っていればtrue。あたり判定が描画関数の中で定義されることに注意。
    this.angle = angle;
    this.magnify = magnify;
};


// 自身を描画。実はポリゴン定義が面倒だったので寸法が設定値よりだいぶ大きくなってしまうが、まあ面倒なので放置。

IconLeaf.prototype.drawme = function (ctx) {
    var MG = 0;
    var RD = this.H*0.2;
    ctx.lineCap = "butt";

    ctx.save(); // 事前セーブ
    this.offset(ctx, this.angle, this.magnify, this._x-this.W/2, this._y-this.H/2, this.W, this.H, this._x+this.W/2, this._y+this.H/2);

    // 外枠（当たり判定：図形が細いので、パスに判定をつけると操作性が落ちる）
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, this.W/2, 0, -Math.PI*2, true);

    if(this.onMouse == false) {
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+","+ctx.liteCol[1]+"%,"+ctx.liteCol[2]+"%,"+1.0+")";
    } else {
        ctx.fillStyle = "hsla("+ctx.darkCol[0]+","+ctx.darkCol[1]+"%,"+ctx.darkCol[2]+"%, "+1.0+")";
    }
    ctx.lineWidth = 1.0;

    // 外枠。マージンあり
    ctx.beginPath();
    ctx.moveTo(MG+RD, MG);
    ctx.lineTo(this.W-MG-RD, MG);
    ctx.quadraticCurveTo(this.W-MG, MG, this.W-MG, MG+RD); // right top
    ctx.lineTo(this.W-MG, this.H-MG-RD);
    ctx.quadraticCurveTo(this.W-MG, this.H-MG, this.W-MG-RD, this.H-MG); // right bottom
    ctx.lineTo(MG+RD, this.H-MG);
    ctx.quadraticCurveTo(MG, this.H-MG, MG, this.H-MG-RD); // left bottom
    ctx.lineTo(MG, MG+RD);
    ctx.quadraticCurveTo(MG, MG, MG+RD, MG); // left top
    ctx.closePath();
    ctx.fill();

    ctx.restore();
};

// 無回転時の基点を左上とし、任意の場所を中心に回転できる
IconLeaf.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};



// ============================================================================
// scroll lockの状態表示
// ============================================================================


function IconScLock( depth_from, where_x, where_y, W, H ) {
    this.depth_from = depth_from; // まだサポートしていないが、描画用の深度の起点
    this._x = where_x; // どこに表示するかはDOMに依存して決まるので、この部分の扱いはひとまず保留
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.onMouse = false; // ボタンにマウスが被っていればtrue。あたり判定が描画関数の中で定義されることに注意。
    this.lock = 0;
    this.angle = 0;
    this.magnify = 1;
    this.radius = 10;
};


// 状態更新メソッド。
IconScLock.prototype.setStatus = function(lock) {
    this.lock = lock;
}


// 自身を描画。
IconScLock.prototype.drawme = function (ctx) {

    ctx.save(); // 事前セーブ

    ctx.lineWidth = this.W*0.15;

    if(this.onMouse == false) {
        ctx.strokeStyle = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 0.81)";
        ctx.fillStyle = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 0.81)"; // 塗りつぶしスタイル
    } else {
        ctx.strokeStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)";
        ctx.fillStyle = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)"; // 塗りつぶしスタイル
    }

    this.offset(ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H);

    // 外枠（当たり判定：図形が細いので、パスに判定をつけると操作性が落ちる）
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, this.H);
    ctx.lineTo(this.W, this.H);
    ctx.lineTo(this.W, 0);
    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。

    if (this.lock > 0) {

        ctx.lineCap = "round";
        ctx.lineJoin = "mitter";

        ctx.beginPath();
        ctx.moveTo(this.W*0.55, 0);
        ctx.lineTo(this.W*0.075, 0);
        ctx.lineTo(this.W*0.075, this.H*0.45);
        ctx.stroke();

        ctx.beginPath();
        ctx.moveTo(this.W*0.45, this.H);
        ctx.lineTo(this.W*0.95, this.H);
        ctx.lineTo(this.W*0.95, this.H*0.55);
        ctx.stroke();

        ctx.beginPath();
        ctx.moveTo(this.W*0.2, this.H*0.4);
        ctx.lineTo(this.W*0.2, this.H*0.85);
        ctx.lineTo(this.W*0.8, this.H*0.85);
        ctx.lineTo(this.W*0.8, this.H*0.4);
        ctx.fill();

        ctx.lineCap = "butt";
        var rd = this.W*0.15
        ctx.beginPath();
        ctx.moveTo(this.W*0.5+rd, this.H*0.405);
        ctx.lineTo(this.W*0.5+rd, this.H*0.35);
        ctx.arc(this.W*0.5, this.H*0.35, rd, 0, Math.PI*1, true);
        ctx.lineTo(this.W*0.5-rd, this.H*0.405);
        ctx.stroke();

        ctx.font_family='Arial'; // scroll lock であることをプレイヤーに示す。
        ctx.font_size = 9;
        ctx.size_unit = "px";
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
        ctx.fillText( text="Scroll", this.W*0.5, this.H*0.65, this.W);
    } else {

        ctx.lineCap = "round";
        ctx.lineJoin = "mitter";

        ctx.beginPath();
        ctx.moveTo(this.W*0.2, this.H*0.3);
        ctx.lineTo(this.W*0.5, 0);
        ctx.lineTo(this.W*0.8, this.H*0.3);
        ctx.stroke();

        ctx.beginPath();
        ctx.moveTo(this.W*0.2, this.H*0.7);
        ctx.lineTo(this.W*0.5, this.H);
        ctx.lineTo(this.W*0.8, this.H*0.7);
        ctx.stroke();

        ctx.beginPath();
        ctx.arc(this.W*0.5, this.W*0.5, this.W*0.2, 0, Math.PI*2, true);
        ctx.fill();

        ctx.font_family='Arial';
        ctx.font_size = 9;
        ctx.size_unit = "px";
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.fillStyle = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";
        ctx.fillText( text="Scroll", this.W*0.5, this.H*0.5, this.W);
    }

    ctx.restore();
};


// グローバル座標上のキャラクター位置を指定位置に移動。
IconScLock.prototype.offset = function(ctx, angle, magnify, where_x, where_y, W_CTX, H_CTX) {
    ctx.translate(where_x , where_y );
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.translate(W_CTX/2, H_CTX/2);
    ctx.rotate( angle /180 * Math.PI);
    ctx.translate(-W_CTX/2, -H_CTX/2);
};
