// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// このファイルには、主にゲーム本画面用のベース要素のデザイン定義を入れる。
// レスポンシブな部品配置にすることを考えると、基本的にはグローバル座標系で配置する部品だけをここに置くべき。
// ============================================================================


// Gamebaseクラスを自作する。こいつはctxと同じ場所に同じサイズで最初に描画され、ゲーム全体の土台を作る（パーツは含まず、地色だけ）。
// なお、setStatusメソッドも今のところ用意していない。ゲーム内状態に応じて土台の色や場所が変わることはないだろうし、
// スマートフォン等で画面回転する場合の対応はhtmlレベルで済ませるほうが明快なので。


function Gamebase(depth, fromX, fromY, toX, toY, image, repetition) {
    this.depth = depth;
    this._x = fromX; // 定数。本来はconstを付けたいが、IE10以前で非対応なので諦める
    this._y = fromY; //
    this.W = toX - fromX;
    this.H = toY - fromY;
    this.init_angle = 0;
    this.init_magnify = 1.0;
    this.image = image; // 定義済みの img, canvas, video 要素のいずれか
    this.repetition = repetition; // "repeat", "repeat-x", "repeat-y", "no-repeat"
};


Gamebase.prototype.offset = function(ctx, rad, magnify, where_x, where_y) {
    ctx.translate(where_x, where_y); // 左上起点なのでこれでいい
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -rad ); // 左上を起点として半時計回り
};

// 自身を描画するメソッド。現在は不使用だが、背景画像によるパターン、ベタ塗り、グラデーションの各手法のリファレンスに残しておく
Gamebase.prototype.drawme = function (ctx) {

    var BASE_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var BASE_STROKE = "rgba(0, 0, 5, 1)";
    var MG = 1; // Margin
    var RD = 2; // radius

    var ptrn = ctx.createPattern(this.image, this.repetition); // パターン作成はctxを指定する必要がある。

    var base_grad  = ctx.createLinearGradient(x0=0, y0=0, x1=0, y1=this.H); // 開始地点 (x0, y0) 終了地点 (x1, y1)
    // グラデーション 黒・銀・銀・黒
    base_grad.addColorStop(0, 'rgba(20, 20, 20, 1)');
    base_grad.addColorStop(0.025, 'rgba(223, 225, 233, 1)');
    base_grad.addColorStop(0.975, 'rgba(242, 245, 243, 0)');
    base_grad.addColorStop(1, 'rgba(20, 20, 20, 1)');

    ctx.save(); // 事前セーブ

    this.offset(ctx, this.angle, this.magnify, this._x, this._y);
    ctx.beginPath(); // 外枠。マージンあり
    ctx.moveTo(MG+RD, MG);
    ctx.lineTo(this.W-MG-RD, MG);
    ctx.quadraticCurveTo(this.W-MG, MG, this.W-MG, MG+RD); // right top
    ctx.lineTo(this.W-MG, this.H-MG-RD);
    ctx.quadraticCurveTo(this.W-MG, this.H-MG, this.W-MG-RD, this.H-MG); // right bottom
    ctx.lineTo(MG+RD, this.H-MG);
    ctx.quadraticCurveTo(MG, this.H-MG, MG, this.H-MG-RD); // left bottom
    ctx.lineTo(MG, MG+RD);
    ctx.quadraticCurveTo(MG, MG, MG+RD, MG); // left top
    ctx.closePath();

    ctx.globalAlpha = 1;
//    ctx.fillStyle = BASE_FILL;
    ctx.fillStyle = base_grad;
    ctx.fill(); // 先に地色を塗ってから、
    ctx.globalAlpha = 0.55;
    ctx.fillStyle = ptrn;
    ctx.fill(); // 上にテクスチャーの画像を乗せる（alpha channelを持つ画像）。

    ctx.restore();
}


// ============================================================================
// 液晶パネル
// Displayクラスは液晶パネルのフレーム外枠および背景を提供する。
// 表示内容はこいつの上に重ねて描画する。ローカル座標系や新たなコンテキストを作成する訳ではない。
// ============================================================================


// Displayクラスを自作する。
function Display(depth, fromX, fromY, toX, toY, angle, magnify, image, repetition) {
    this.depth = depth;
    this._x = fromX; // 定数。本来はconstを付けたいが、IE10以前で非対応なので諦める
    this._y = fromY; //
    this.W = toX - fromX;
    this.H = toY - fromY;
    this.angle = angle;
    this.magnify = magnify;
    this.image = image; // 定義済みの img, canvas, video 要素のいずれか
    this.repetition = repetition; // "repeat", "repeat-x", "repeat-y", "no-repeat"
};


Display.prototype.offset = function(ctx, rad, magnify, where_x, where_y) {
    ctx.translate(where_x, where_y); // 左上起点なのでこれでいい
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -rad ); // 左上を起点として半時計回り
};


// 毎フレーム、キャラクター自身を描画させるメソッド。
Display.prototype.drawme = function (ctx) {

    // 定数の初期化
    var BASE_FILL = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)";
    var BASE_STROKE = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.darkCol[2]+"%, 1)";
    var MG = 0;
    var RD = 15;
    var ptrn = ctx.createPattern(this.image, this.repetition); // パターン作成はctxを指定する必要がある。

    ctx.save(); // 事前セーブ

    this.offset(ctx, this.angle, this.magnify, this._x, this._y);
    ctx.beginPath(); // 外枠。マージンあり
    ctx.moveTo(MG+RD, MG);
    ctx.lineTo(this.W-MG-RD, MG);
    ctx.quadraticCurveTo(this.W-MG, MG, this.W-MG, MG+RD); // right top
    ctx.lineTo(this.W-MG, this.H-MG-RD);
    ctx.quadraticCurveTo(this.W-MG, this.H-MG, this.W-MG-RD, this.H-MG); // right bottom
    ctx.lineTo(MG+RD, this.H-MG);
    ctx.quadraticCurveTo(MG, this.H-MG, MG, this.H-MG-RD); // left bottom
    ctx.lineTo(MG, MG+RD);
    ctx.quadraticCurveTo(MG, MG, MG+RD, MG); // left top
    ctx.closePath();

    ctx.globalAlpha = 1;
    ctx.strokeStyle = BASE_STROKE; // ディスプレイの縁
    ctx.lineWidth = 3;
    ctx.stroke();
    ctx.fillStyle = BASE_FILL; // 液晶画面の地の色
    ctx.fill();
    ctx.globalAlpha = 0.15;
    ctx.fillStyle = ptrn; // 液晶の保護フィルムの模様。あまり正確に再現する意図はない。
    ctx.fill();

    ctx.restore(); // 事後ロード。これで拡大、オフセット、回転は全てリセットされ、後続の描画処理には影響しない。
}


// ============================================================================
// 上記の中央液晶パネル上にゼロフレーム目で表示させるタイトル。
// 現在は単なるタイトル文字の定義だが、mainルーチンからより多くの機能を移動させるべきであろう。
// ============================================================================


function Main00(depth, fromX, fromY, toX, toY, font_family, font_size, size_unit) {
    this.depth = depth;
    this._x = fromX;
    this._y = fromY;
    this.W = toX - fromX;
    this.H = toY - fromY;
    this.angle = -Math.PI/2 ;
    this.magnify = 1.0;
    this.font_family=font_family;
    this.font_size = font_size;
    this.size_unit = size_unit;
};


Main00.prototype.drawme = function (ctx) {

    var TITLE_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)";

    ctx.save();

    // labels
    ctx.textAlign = "center";
    ctx.fillStyle = TITLE_FILL;
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    ctx.fillText("KABURI CANVAS", this._x + this.W/2, this._y + this.H/2 + this.font_size*2.0 );
    ctx.font = this.font_size/2 + this.size_unit + " " + this.font_family;
    ctx.fillText("Version 1.0", this._x + this.W/2, this._y + this.H/2 + this.font_size*3.0 );

    ctx.restore();
}


// ============================================================================
// ゲームオーバー （status 101） 時にタイトル画面に表示させる。
// ============================================================================


function Main101(depth, fromX, fromY, toX, toY, font_family, font_size, size_unit) {
    this.depth = depth;
    this._x = fromX;
    this._y = fromY; //
    this.W = toX - fromX;
    this.H = toY - fromY;
    this.angle = -Math.PI/2 ;
    this.magnify = 1.0;
    this.font_family=font_family;
    this.font_size = font_size;
    this.size_unit = size_unit;

    this.show_leaf = new IconLeaf(  depth_from=20001, angle=0, magnify=1.0,
                                    where_x=this._x+this.W/2-150, where_y=this._y+this.H/2+30, W=50, H=50)
};


Main101.prototype.setStatus = function (cleartime, leaf, leaf_sum, enemies, kaburi) {
    this.cf = cleartime;
    this.leaf = leaf;
    this.leaf_sum = leaf_sum;
    this.enemies = enemies;
    for(var i = 0; i < Math.min(this.enemies.cumul, this.enemies.popul_max); i=(i+1)|0) {
        this.enemies.ary[i].magnify = 0.940;
    }
    this.kaburi = kaburi;
}


Main101.prototype.drawme = function (ctx) {
    var TITLE_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var WHITE_FILL = "rgba(250, 250, 255, 1)"; // 塗りつぶし。
    var CAUSE_FILL = "rgba(250, 250, 5, 1)"; // 塗りつぶし。ゲームオーバーの原因となった変数に使う。
    var TITLE_STROKE = "rgba(0, 0, 5, 1)"; // 線色。

    ctx.save();

    ctx.lineWidth = 2;
    ctx.strokeStyle = "rgba(0, 0, 5, 1)";
    ctx.fillStyle = "rgba(5, 5, 5, 1)";
    ctx.beginPath(); // 葉っぱの下の色を締めるために、黒系で下塗りしている。
    ctx.moveTo(LEAF_FROM_X, LEAF_FROM_Y);
    ctx.lineTo(LEAF_FROM_X, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y);
    ctx.closePath();
    ctx.stroke();
    ctx.fill();

    ctx.save();
    this.leaf.offset( ctx, this.leaf.angle, this.leaf.magnify, this.leaf._x, this.leaf._y,
                      this.leaf.W, this.leaf.H, this.leaf.cx, this.leaf.cy);
    for(var i = 0; i < this.leaf.ny; i=(i+1)|0) {
        for(var h = 0; h < this.leaf.nx; h=(h+1)|0) {
            this.leaf.ary[h][i].drawme(ctx); // leaflet の各セルを描画する処理。
        }
    }
    ctx.restore();

    ctx.fillStyle = "rgba(5, 5, 10, 0.8)";
    ctx.beginPath(); // 葉っぱの上にゲーム結果の表示をオーバーレイするため、もう一度下塗りする。
    ctx.moveTo(LEAF_FROM_X, LEAF_FROM_Y);
    ctx.lineTo(LEAF_FROM_X, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y);
    ctx.closePath();
    ctx.fill();

    ctx.shadowBlur = 6;
    ctx.shadowColor = "rgba(5, 5, 5, 1)";

    // ここから、スコア表示部
    var sint = 75;
    renderHadaniLineup( ctx=ctx, angle=Math.PI*0, magnify=1.0,
                        fromX=LEAF_FROM_X+LEAF_W/2-sint*2, fromY=LEAF_FROM_Y+LEAF_H/2-70,
                        intervalX=sint, intervalY=0, cx=40*2, cy=0);
    ctx.textAlign = "center";
    ctx.fillStyle = WHITE_FILL;
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    for(var i = 0; i < 5; i=(i+1)|0) {
        ctx.fillText(this.enemies.eaten_cumul_array[i], LEAF_FROM_X+LEAF_W/2+2+sint*(i-2), LEAF_FROM_Y+LEAF_H/2-100 );
    }

    this.show_leaf.drawme(ctx); // 葉っぱのスコア
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    ctx.textAlign = "left";
    if (this.leaf_sum <= 10) {
        ctx.fillStyle = CAUSE_FILL; // ゲームボーバーの至近要因となった判定基準のスコア部を黄色く塗るための条件分岐。
    } else {
        ctx.fillStyle = WHITE_FILL;
    }
    ctx.fillText(this.leaf_sum, LEAF_FROM_X+LEAF_W/2-100, LEAF_FROM_Y+LEAF_H/2+45 );

    this.kaburi.drawClipping( ctx, fromX=LEAF_FROM_X, fromY=LEAF_FROM_Y,
                              toX=(LEAF_FROM_X+LEAF_W), toY=(LEAF_FROM_Y+LEAF_H)); // カブリダニ
    ctx.fillStyle = WHITE_FILL;
    ctx.font = this.font_size*0.75 + this.size_unit + " " + this.font_family;
    ctx.fillText("EXP: " + this.kaburi.experience, LEAF_FROM_X+LEAF_W/2+65, LEAF_FROM_Y+LEAF_H/2+20 );

    if (this.kaburi.life <= 100) {
        ctx.fillStyle = CAUSE_FILL;
    } else {
        ctx.fillStyle = WHITE_FILL;
    }
    ctx.fillText("LIFE:" + Math.floor(this.kaburi.life/this.kaburi.LIFE_MAX*100) + "%",
                 LEAF_FROM_X+LEAF_W/2+65, LEAF_FROM_Y+LEAF_H/2+60 );

    ctx.textAlign = "center";
    if (this.cf >= MAXTIME) {
        ctx.fillStyle = CAUSE_FILL;
    } else {
        ctx.fillStyle = WHITE_FILL;
    }
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    ctx.fillText("GAME OVER: "+ this.cf +" frames", this._x+this.W/2, this._y+this.font_size*2 );

    ctx.fillStyle = WHITE_FILL;
    ctx.fillText("KABURI CANVAS", this._x+this.W/2, this._y+this.H/2+this.font_size*3.5 );
    ctx.font = this.font_size/2 + this.size_unit + " " + this.font_family;
    ctx.fillText("Version 1.0", this._x+this.W/2, this._y+this.H/2+this.font_size*4.5 );

    // ここに、スコア送信機能をつける。まずハンドルネームの入力用テキストフィールドか。だが故あって未実装。

    ctx.restore();
}



// stage == 101 ゲームオーバー後の成績表示
// stage == 201 ゲームクリア後の結果表示および送信画面

// クリア画面（まだ作りこみは十分ではない。通信系の機能は外してある）
// 現在のゲームステータス（Cleared / Game over）
// 現在の葉っぱのパッチ残数表示
// 食べたハダニの齢別個体数
// 必要ならばデータベースアクセスし、ランキング表示。名前入力欄を設ける。
// プレイの詳細ログをsudori.infoへ送るか？チェックボックスに標準でチェック入、送るデータの内訳はポップアップ
// 現在の設定でリトライ
// ゲームエンド
// スコアを見る

// ============================================================================
// ゲームクリア （status 201） 時にタイトル画面に表示させる。
// ============================================================================

function Main201( depth, fromX, fromY, toX, toY, font_family, font_size, size_unit) {
    this.depth = depth;
    this._x = fromX;
    this._y = fromY;
    this.W = toX - fromX;
    this.H = toY - fromY;
    this.angle = -Math.PI/2 ;
    this.magnify = 1.0;
    this.font_family=font_family;
    this.font_size = font_size;
    this.size_unit = size_unit;
    this.show_leaf = new IconLeaf(  depth_from=20001, angle=0, magnify=1.0,
                                    where_x=this._x+this.W/2-160, where_y=this._y+this.H/2+30, W=50, H=50)
};


Main201.prototype.setStatus = function (cleartime, leaf, leaf_sum, enemies, kaburi) {
    this.cf = cleartime;
    this.leaf = leaf;
    this.leaf_sum = leaf_sum;
    this.enemies = enemies;
    this.kaburi = kaburi;
}


Main201.prototype.drawme = function (ctx) {
    var TITLE_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.liteCol[2]+"%, 1)"; // 塗りつぶし。
    var WHITE_FILL = "rgba(250, 250, 255, 1)"; // 塗りつぶし。
    var TITLE_STROKE = "rgba(0, 0, 5, 1)"; // 線色。

    ctx.save(); // 事前セーブ

    ctx.lineWidth = 2;
    ctx.strokeStyle = "rgba(0, 0, 5, 1)";
    ctx.fillStyle = "rgba(5, 5, 5, 1)";
    ctx.beginPath(); // 葉っぱの下の色を締めるために、黒系で下塗りしている。
    ctx.moveTo(LEAF_FROM_X, LEAF_FROM_Y);
    ctx.lineTo(LEAF_FROM_X, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y);
    ctx.closePath();
    ctx.stroke();
    ctx.fill();

    ctx.save();
    this.leaf.offset( ctx, this.leaf.angle, this.leaf.magnify, this.leaf._x, this.leaf._y,
                      this.leaf.W, this.leaf.H, this.leaf.cx, this.leaf.cy);
    for(var i = 0; i < this.leaf.ny; i=(i+1)|0) {
        for(var h = 0; h < this.leaf.nx; h=(h+1)|0) {
            this.leaf.ary[h][i].drawme(ctx); // leaflet の各セルを描画する処理。
//            this.leaf.ary[h][i].drawArc(ctx); // こちらは円で描く。どちらがいいかな？
        }
    }
    ctx.restore();

    ctx.fillStyle = "rgba(5, 5, 10, 0.8)";
    ctx.beginPath(); // 葉っぱの上にゲーム結果の表示をオーバーレイするため、もう一度下塗りする。
    ctx.moveTo(LEAF_FROM_X, LEAF_FROM_Y);
    ctx.lineTo(LEAF_FROM_X, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y+LEAF_H);
    ctx.lineTo(LEAF_FROM_X+LEAF_W, LEAF_FROM_Y);
    ctx.closePath();
    ctx.fill();

    ctx.shadowBlur = 6;
    ctx.shadowColor = "rgba(5, 5, 5, 1)";

    // ここから、スコア表示部
    var sint = 75;
    renderHadaniLineup( ctx=ctx, angle=Math.PI*0, magnify=1.0, fromX=LEAF_FROM_X+LEAF_W/2-sint*2,
                        fromY=LEAF_FROM_Y+LEAF_H/2-70, intervalX=sint, intervalY=0, cx=40*2, cy=0);
    ctx.textAlign = "center";
    ctx.fillStyle = WHITE_FILL;
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    for(var i = 0; i < 5; i=(i+1)|0) {
        ctx.fillText(this.enemies.eaten_cumul_array[i], LEAF_FROM_X+LEAF_W/2+2+sint*(i-2), LEAF_FROM_Y+LEAF_H/2-100);
    }

    this.show_leaf.drawme(ctx); // 葉っぱのスコア
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    ctx.textAlign = "left";
    ctx.fillText(this.leaf_sum, LEAF_FROM_X+LEAF_W/2-110, LEAF_FROM_Y+LEAF_H/2+45 );

//    this.kaburi._x = LEAF_FROM_X+LEAF_W/2+30;
//    this.kaburi._y = LEAF_FROM_Y+LEAF_H/2+25;
    this.kaburi.drawClipping( ctx, fromX=LEAF_FROM_X, fromY=LEAF_FROM_Y,
                              toX=(LEAF_FROM_X+LEAF_W), toY=(LEAF_FROM_Y+LEAF_H)); // カブリダニ
    ctx.font = this.font_size*0.75 + this.size_unit + " " + this.font_family;
    ctx.fillText("EXP: " + this.kaburi.experience, LEAF_FROM_X+LEAF_W/2+75, LEAF_FROM_Y+LEAF_H/2+20 );
    ctx.fillText("LIFE:" + Math.ceil(this.kaburi.life/this.kaburi.LIFE_MAX*100) + "%",
                 LEAF_FROM_X+LEAF_W/2+75, LEAF_FROM_Y+LEAF_H/2+60 );

    ctx.textAlign = "center";
    ctx.font = this.font_size + this.size_unit + " " + this.font_family;
    ctx.fillText("Cleared in "+ this.cf +" frames", this._x+this.W/2, this._y+this.font_size*2 );

    ctx.fillText("KABURI CANVAS", this._x+this.W/2, this._y+this.H/2+this.font_size*3.5 );
    ctx.font = this.font_size/2 + this.size_unit + " " + this.font_family;
    ctx.fillText("Version 1.0", this._x+this.W/2, this._y+this.H/2+this.font_size*4.5 );

    // ここに、スコア送信機能をつける。まずハンドルネームの入力用テキストフィールドか。だが故あって未実装。

    ctx.restore();
}
