// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// キーボードの入力を監視する関数
// ============================================================================

// 以下クラス InputKeyboard() に属するオブジェクト input_key は main.jsで定義され、実際にkaburiの操作で参照されている
// keyinput_bufferはコンピュータのキーの全種類に等しい長さを持つ配列で、番号に対応するキーが押されているかどうかを格納する。

function InputKeys() {

    var kbuffer = new Array();
    this.buffer = kbuffer;

    if (window.addEventListener) {
        document.addEventListener("keydown", function(e) {
            kbuffer[e.keyCode] = true;
        }, false);
        document.addEventListener("keyup", function(e) {
            kbuffer[e.keyCode] = false;
        }, false);
        window.addEventListener("blur", function() {
            kbuffer.length = 0;
        }, false);
    } else if (window.attachEvent) {
        document.attachEvent("onkeydown", function(e) {
            kbuffer[e.keyCode] = true;
        }, false);
        document.attachEvent("onkeyup", function(e) {
            kbuffer[e.keyCode] = false;
        }, false);
        window.attachEvent("onblur", function() {
            kbuffer.length = 0;
        }, false);
    }
};

// キーコードを指定してバッファにアクセスし、該当するキーの押下の有無を取得するメソッド
InputKeys.prototype.isDown = function(key_code) {
    ans = false;
    if(this.buffer[key_code] == true) {
        ans = true;
    }

    return ans;
};



// - event --------------------------------------------------------------------

// スクロールの禁止
// http://output.jsbin.com/xatidu/4/
// disableScroll() の関数を実行させれば、スクロールが止まる
// enableScroll() でスクロール禁止の解除

// 実際には、この関数はmain内ではなく、htmlの該当の要素に onfocus="disableScroll();" onblur="enableScroll();" として指定。

// 実装はしていないものの、たとえばスペースバーを使ってゲームの進行を一時停止するとかいった使い方もありうる。
// その場合、基本的には以下の preventDefaultForScrollKeys(e) はそのままでよくて、その代わり別の場所でキー状態を調査させる。

// なお、スマホだとスワイプがもたついてしまうという問題が発覚。
// touch有効かつキーボード入力無しの環境では、このpreventDefault機能が有効にならないようにしてはどうか？
// 問題はどうやって「環境にキーボードが無い」ことを検出するか

function preventDefault(e) {
    if (e == false) {
        e = window.event;
    }
    if (e.preventDefault == true) {
        e.preventDefault();
    }
    e.returnValue = false; // これはIE系のための処理で、実行せんとするイベントに強制的にfalseを吐かせて止めている
}

function preventDefaultForScrollKeys(e) {
    // left: 37, up: 38, right: 39, down: 40,
    // spacebar: 32, pageup: 33, pagedown: 34, end: 35, home: 36
    var keys = {32:1, 33:1, 34:1, 35:1, 36:1, 37:1, 38:1, 39:1, 40:1}; // コロンは連想配列。

    if (keys[e.keyCode]) {
        preventDefault(e);
        return false;
    }
}


// 実際にスクロールを禁止する手続き
function disableScroll() {
    if (window.addEventListener == true) { // older FF
        window.addEventListener('DOMMouseScroll', preventDefault, false);
    }
    window.onwheel = preventDefault; // modern standard
    window.onmousewheel = document.onmousewheel = preventDefault; // older browsers, IE
    window.ontouchmove  = preventDefault; // mobile
    // スワイプは可能にしたい。こまったな。
    // https://blog.mobiscroll.com/working-with-touch-events/
    document.onkeydown  = preventDefaultForScrollKeys; // 上で定義した、キーボードの上下左右を受け付けなくする操作
}


// スクロール禁止の解除
// いったん登録された要素についてpreventDefaultを解呪する方法がないので、呪いの掛かった要素自体を破棄することで対処している。

function enableScroll() {
    if (window.removeEventListener == true) {
        window.removeEventListener('DOMMouseScroll', preventDefault, false);
    }
    window.onmousewheel = document.onmousewheel = null;
    window.onwheel = null;
    window.ontouchmove = null;
    document.onkeydown = null;
}



// ============================================================================
/*
ここから、スマホで自機の方向を制御するためのポインター（pointing stick）を作る。
候補案
1) 十字キーを描画する。もっとも楽だが、場所を取る。スマホでは無理？
2) カブリに直接触る、あるいはハダニをタップすると自動追尾する
3) touchmoveだけに反応して指の動いた方向へ一定距離動くデバイスを作る

ひとまず、ジョイスティックのようなものを作ってみた。
*/
// ============================================================================


function Pointer(where_x, where_y, w, h, radius, text_array, power, angle, magnify) {
    this.angle = angle;
    this.magnify = magnify;
    this._x = where_x;
    this._y = where_y;
    this.W = w; // w, h が本体の幅と高さ
    this.H = h;
    this.radius = radius; // radiusは箱の角丸の半径。
    this.onMouse = false; // ボタンにマウスが被っていればtrue
    this.text = text_array;
    this.power = power;
    this.rho = 0; //  ポインター状態の極座標系における半径＝中心（中立状態）からのずれの量。0 <= rho <= 1 で定義
    this.psi = -1; // ポインター状態の極座標系における角度。初期化は-1、更新後は0 <= psi <= Math.PI*2
    this.react_area = 0.125; // この値がゼロに近いほど、マウスや指が遠くにあっても入力を拾う。
}


Pointer.prototype.setStatus = function(ctx, texist, mousedown) {
    this.rho = 0; //  ポインターリセット
    this.psi = -1;
    var mx = ctx.mouse_layer_x;
    var my = ctx.mouse_layer_y;
    var rx = ctx.mouse_layer_x-(this._x+this.W/2);
    var ry = ctx.mouse_layer_y-(this._y+this.H/2);
    if (texist > 0 || mousedown == true) { // 正しくは、mouse button の押下にも反応させたい。
        if ((rx*rx+ry*ry)*this.react_area <= this.radius*this.radius) {
            this.rho = Math.min(Math.pow((rx*rx+ry*ry)/(this.radius*this.radius), 0.5), 0.5);
            this.psi = Math.atan2(ry, rx); // アークタンジェント。yが第一引数。
            console.log(this.rho);
        }
    }
}

// 自身を描画。
Pointer.prototype.drawme = function (ctx) {
    ctx.save();
    this.offset( ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H, this._x+this.W/2, this._y+this.H/2 );
    this.drawBody(ctx=ctx);
    this.drawPict(ctx=ctx);
    this.drawText(ctx=ctx);
    ctx.restore();
};


// 無回転時の基点を左上とし、任意の場所を中心に回転できる
Pointer.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};


Pointer.prototype.drawBody = function (ctx) {
    // パス定義を繰り返す。1回目はstrokeやfillで本体の描画を行わなず、内外判定だけに使う
    this.renderBodyShape(ctx);
    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。
    this.renderBodyShape(ctx);
    var FILL0 = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)"; // 塗りつぶし。
    var FILL1 = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)"; // 塗りつぶし。
    if(this.onMouse == false) {
        ctx.fillStyle = FILL0;
    } else {
        ctx.fillStyle = FILL1;
    }

    var W = this.W;
    var H = this.H;
    var RD = this.radius;
    var PSI = this.psi;
    var RHO = this.rho;
    var grd=ctx.createRadialGradient(  W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), 0,
                                       W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), RD*1.5);
    grd.addColorStop(0, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)");
    grd.addColorStop(0.45, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)");
    grd.addColorStop(0.91, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+10+"%, 1)");
    ctx.fillStyle=grd;

    ctx.fill();
    ctx.strokeStyle = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 0.7)";
    ctx.lineWidth = 4;
    ctx.stroke();
};


Pointer.prototype.renderBodyShape = function (ctx) {
    var W = this.W;
    var H = this.H;
    var RD = this.radius

    ctx.beginPath();
    if (RD < Math.min(W, H)/2) {
        ctx.arc(W/2, RD, RD, -Math.PI*0.25, -Math.PI*0.75, true);
        ctx.arc(RD, H/2, RD, -Math.PI*0.75, -Math.PI*1.25, true);
        ctx.arc(W/2, H-RD, RD, -Math.PI*1.25, -Math.PI*1.75, true);
        ctx.arc(W-RD, H/2, RD, -Math.PI*1.75, -Math.PI*2.25, true);

    } else {
        ctx.arc(W/2, H/2, RD, 0, -Math.PI*2, true);
    }
    ctx.closePath();
};


// pixel ratio が 1.0 でないとき、shadowを飛ばすと戻ってこないというバグがあったので修正した。
// どうやら shadowOffsetX shadowOffsetY の両メソッドにはdevicePixelRatioが反映されないのですね。想定外だ。
// 2016/06/25 ところがぎっちょん、viewport width="960px" などとして直接数字を打込んだ場合、この方法は逆に
// ズレを生んでしまう。つまり、何も考えない初期状態で良い。

Pointer.prototype.drawPict = function (ctx, fill, stroke) {
    ctx.save();
    ctx.shadowOffsetX = 1;
    ctx.shadowOffsetY = 1;
    ctx.shadowColor = "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 0.90)";
    ctx.shadowBlur = 10; // ぼかし量
    ctx.fillStyle = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.semiCol[2]+"%, 1)";

    var W = this.W;
    var H = this.H;
    var RD = this.radius;
    var PSI = this.psi;
    var RHO = this.rho;
    var grd=ctx.createRadialGradient(  W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), 0,
                                       W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), RD/2);
    grd.addColorStop(0.0, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.275, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.30, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.50, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.525, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.55, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.575, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.70, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.725, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.75, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.775, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.85, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(1, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.semiCol[2]+"%, 1)");
    ctx.fillStyle=grd;

    this.renderPictShape(ctx);
    ctx.fill(); // こいつがないと影自体描画されない
    ctx.restore(); // restoreしないと後続のtextにまでclipが影響してしまうので注意。
};


Pointer.prototype.renderPictShape = function (ctx) {
    var W = this.W;
    var H = this.H;
    var RD = this.radius
    ctx.beginPath();
    ctx.arc(W/2+RD*this.rho*Math.cos(this.psi), H/2+RD*this.rho*Math.sin(this.psi), RD/2, 0, -Math.PI*2, true);
    ctx.closePath();
};


// 文字の描画処理。
Pointer.prototype.drawText = function (ctx) {

    ctx.save();
    if (this.onMouse == true) {
        ctx.fillStyle = "rgba(50, 150, 255, 1.0)"; // blue?
        ctx.shadowOffsetX = -1;
        ctx.shadowOffsetY = -1;
        ctx.shadowColor = "rgba(0, 0, 5, 0.60)";
        ctx.shadowBlur = 1; // ぼかし量
    } else {
        ctx.fillStyle = "rgba(155, 155, 155, 1)";
        ctx.shadowOffsetX = -1;
        ctx.shadowOffsetY = -1;
        ctx.shadowColor = "rgba(0, 0, 5, 0.51)";
        ctx.shadowBlur = 1; // ぼかし量
    }
    ctx.font= '24pt Arial';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText( this.text[power], this.W/2, this.H/2, this.W); // テキストは形状指定と同時に描画される
    ctx.restore();
}



// 外枠（オプション）
Pointer.prototype.drawOuter = function (ctx) {
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, this.H);
    ctx.lineTo(this.W, this.H);
    ctx.lineTo(this.W, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.strokeStyle = "rgba(5, 5, 5, 1)";
    ctx.stroke(); // strokeで輪郭。
};


// ボディ中央マーカー（オプション）
Pointer.prototype.drawCenter = function (ctx, fill, stroke) {
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, this.H/20, 0, Math.PI*2, false);
    ctx.fillStyle = fill;
    ctx.fill();
};



// ============================================================================
/*
上記のジョイスティック方式は、正確な方向制御にかなりの熟練が必要な事が判明。
特に、一定方向へ移動中、急に方向を変えたくなった場合、いったん戻さねばならない。
このとき物理的な復元力が働かないため、指のホームポジションが分からない。

というわけでマイナーチェンジとして、中心方向に少し指が動くと自動でホームポジションに戻るようにしてみた。
*/
// ============================================================================


function Dpad(where_x, where_y, w, h, radius, text_array, power, angle, magnify) {
    this.angle = angle;
    this.magnify = magnify;
    this._x = where_x;
    this._y = where_y;
    this.W = w; // w, h が本体の幅と高さ
    this.H = h;
    this.radius = radius; // radiusは箱の角丸の半径。
    this.onMouse = false; // ボタンにマウスが被っていればtrue
    this.text = text_array;
    this.power = power;
    this.rho = 0; //  ポインター状態の極座標系における半径＝中心（中立状態）からのずれの量。0 <= rho <= 1 で定義
    this.psi = -1; // ポインター状態の極座標系における角度。初期化は-1、更新後は0 <= psi <= Math.PI*2
    this.psi_tmp = -1;
    this.rho_former = 1;

    this.react_area = 0.3; // この値がゼロに近いほど、マウスや指が遠くにあっても入力を拾う。
}


Dpad.prototype.setStatus = function(ctx, texist, mousedown) {

    this.rho_former = this.rho*0.925;
    this.rho = 0; //  ポインターリセット
    this.psi = -1; // ポインターの原点からの極座標。rhoが距離でpsiが角度。
    var mx = ctx.mouse_layer_x;
    var my = ctx.mouse_layer_y;
    var rx = ctx.mouse_layer_x-(this._x+this.W/2);
    var ry = ctx.mouse_layer_y-(this._y+this.H/2);
    if (texist > 0 || mousedown == true) { // 正しくは、mouse button の押下にも反応させたい。
        if ((rx*rx+ry*ry)*this.react_area <= this.radius*this.radius) {
            this.rho = Math.min(Math.pow((rx*rx+ry*ry)/(this.radius*this.radius), 0.15), 1.0);
            this.psi_tmp = Math.atan2(ry, rx)/Math.PI; // アークタンジェント。yが第一引数。
            this.psi = this.psi_tmp*Math.PI; // 8方向への限定無し
            this.psi = Math.round(this.psi_tmp*4)*Math.PI/4; // 8方向にしか動けない
            console.log("rho, rho_former, psi:"+ this.rho + " " + this.rho_former + " " + this.psi);
            if (this.rho <= 0.72 || this.rho <= this.rho_former) {
                this.rho = 0;
            }
        }
    }
}

// 自身を描画。
Dpad.prototype.drawme = function (ctx) {
    ctx.save();
    this.offset( ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H, this._x+this.W/2, this._y+this.H/2 );
    this.drawBody(ctx=ctx);
    this.drawPict(ctx=ctx);
    this.drawText(ctx=ctx);
    ctx.restore();
};


// 無回転時の基点を左上とし、任意の場所を中心に回転できる
Dpad.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};


Dpad.prototype.drawBody = function (ctx) {
    // パス定義を繰り返す。1回目はstrokeやfillで本体の描画を行わなず、内外判定だけに使う
    this.renderBodyShape(ctx);
    this.onMouse = ctx.isPointInPath(ctx.mouse_layer_x, ctx.mouse_layer_y);    // ここに内外判定。
    this.renderBodyShape(ctx);
    var FILL0 = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 1)"; // 塗りつぶし。
    var FILL1 = "hsla("+ctx.semiCol[0]+", "+ctx.semiCol[1]+"%, "+ctx.semiCol[2]+"%, 1)"; // 塗りつぶし。
    if(this.onMouse == false) {
        ctx.fillStyle = FILL0;
    } else {
        ctx.fillStyle = FILL1;
    }

    var W = this.W;
    var H = this.H;
    var RD = this.radius;
    var PSI = this.psi;
    var RHO = this.rho;
    var grd=ctx.createRadialGradient(  W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), 0,
                                       W/2+RD*RHO*Math.cos(PSI), H/2+RD*RHO*Math.sin(PSI), RD*1.5);
    grd.addColorStop(0, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.liteCol[2]+"%, 1)");
    grd.addColorStop(0.34, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)");
    grd.addColorStop(0.35, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)");
    grd.addColorStop(0.36, "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+10+"%, 1)");
    ctx.fillStyle=grd;
    ctx.fill();

    ctx.strokeStyle = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 0.7)";
    ctx.lineWidth = 4;
    ctx.stroke();
};


Dpad.prototype.renderBodyShape = function (ctx) {
    var W = this.W;
    var H = this.H;
    var RD = this.radius
    ctx.beginPath();
    ctx.arc(W/2, H/2, RD, 0, -Math.PI*2, true);
    ctx.closePath();
};


Dpad.prototype.drawPict = function (ctx, fill, stroke) {
    ctx.save();
    ctx.shadowOffsetX = 1;
    ctx.shadowOffsetY = 1;
    ctx.shadowColor = "hsla("+ctx.liteCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 0.90)";
    ctx.shadowBlur = 10; // ぼかし量
    ctx.fillStyle = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.baseCol[2]+"%, 1)";
    for(var i=0; i<8; i=(i+1)|0) {
        this.renderPictShape(ctx);
        ctx.fill(); // こいつがないと影自体描画されない
        ctx.translate(this.W/2, this.H/2);
        ctx.rotate( -Math.PI/4 ); // 中央に移動し、-rotateして、再び左上に戻る。
        ctx.translate(-this.W/2, -this.H/2);
    }
    ctx.restore(); // restoreしないと後続のtextにまでclipが影響してしまうので注意。
};


Dpad.prototype.renderPictShape = function (ctx) {
    var W = this.W;
    var H = this.H;
    var RD = this.radius
    ctx.beginPath();
    ctx.moveTo(W/2, H*0.05);
    ctx.lineTo(W/2-W*0.075, H*0.15);
    ctx.lineTo(W/2+W*0.075, H*0.15);
    ctx.closePath();
};


// 文字の描画処理。
Dpad.prototype.drawText = function (ctx) {

    ctx.save();
    if (this.onMouse == true) {
        ctx.fillStyle = "rgba(50, 150, 255, 1.0)"; // blue?
        ctx.shadowOffsetX = -1;
        ctx.shadowOffsetY = -1;
        ctx.shadowColor = "rgba(0, 0, 5, 0.60)";
        ctx.shadowBlur = 1; // ぼかし量
    } else {
        ctx.fillStyle = "rgba(155, 155, 155, 1)";
        ctx.shadowOffsetX = -1;
        ctx.shadowOffsetY = -1;
        ctx.shadowColor = "rgba(0, 0, 5, 0.51)";
        ctx.shadowBlur = 1; // ぼかし量
    }
    ctx.font= '12pt Arial';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText( this.text[power], this.W/2, this.H*0.9, this.W); // テキストは形状指定と同時に描画される
    ctx.restore();
}



// 外枠（オプション）
Dpad.prototype.drawOuter = function (ctx) {
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, this.H);
    ctx.lineTo(this.W, this.H);
    ctx.lineTo(this.W, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.strokeStyle = "rgba(5, 5, 5, 1)";
    ctx.stroke(); // strokeで輪郭。
};


// ボディ中央マーカー（オプション）
Dpad.prototype.drawCenter = function (ctx, fill, stroke) {
    ctx.beginPath();
    ctx.arc(this.W/2, this.H/2, this.H/20, 0, Math.PI*2, false);
    ctx.fillStyle = fill;
    ctx.fill();
};

