// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// Leafクラス：パッチが集合した1枚の葉に相当する。
// Leafletクラス：パッチ1枚に相当する。葉を構成する細胞と考えればいい。
// ============================================================================

// Leafクラスを自作する。これはパッチの集合体で、ゲーム中に1回しか使用されない。
function Leaf(depth, fromX, fromY, toX, toY, nx, ny, angle, magnify, cx, cy, image) {
    this.depth = depth;
    this._x = fromX;
    this._y = fromY;
    this.W = toX-fromX; // w, h は 葉を構成するleaflet の幅と高さ。W, H は葉ぜんたいの高さ
    this.H = toY-fromY;
    this.nx = nx;
    this.ny = ny;
    this.w = Math.floor(this.W/nx);
    this.h = Math.floor(this.H/ny);
    this.angle = angle;
    this.magnify = magnify;
    this.cx = cx; // cx, cyは回転の中心であり、拡大縮小の基点でもある
    this.cy = cy;
    this.image = image; // image は Image クラスオブジェクトで与える。

    // leafletを配列要素として格納。見ての通り配列は「上からi番目、左からj番目」の位置の葉パッチを定義している。
    this.ary = new Array(this.ny);
    for(var i = 0; i < this.ny; i=(i+1)|0) {
        var aryx = new Array(this.nx);
        for(var h = 0; h < this.nx; h=(h+1)|0) {
            aryx[h] = new Leaflet(i*this.w, h*this.h, this.w, this.h, angle, magnify);
        }
        this.ary[i] = aryx;
    }
};

// サイズと描画位置を再定義したい場合に使う。
// magnifyは新たな描画スケール。縦横比の調整は今の所必要ないだろうから未サポート。描画位置は「左上」から。

Leaf.prototype.rescale = function(where_x, where_y, angle, magnify, cx, cy) {
    this._x = where_x;
    this._y = where_y;
    this.angle = angle;
    this.magnify = magnify;
    this.cx = cx; // cx, cyは回転の中心であり、拡大縮小の基点でもある
    this.cy = cy;
};


// 毎フレーム、葉の状態を更新する処理は、実行速度の兼ね合いでハダニの更新処理の際に弄ってしまうので、無い。


// 与えられた座標に対応するパッチを返す関数。
Leaf.prototype.returnPatch = function (pointx, pointy) {
    var ans = new Array(-1,-1);
    if (pointx >= this._x && pointx <= this._x + this.W) {
        if (pointy >= this._y && pointy <= this._y + this.H) {
            var x=pointx - this._x;
            var y=pointy - this._y;
            ans[0] = Math.floor(x / this.w);
            ans[1] = Math.floor(y / this.h);
        }
    }
    return ans ;
}


// 現在の残存栄養量を返す関数。問題は、ハダニが食べ残してポイントがわずかに残っているパッチの扱い。
// 引数 threshold を用い、一定以上の栄養が残っている葉だけを加算対象にする。

Leaf.prototype.sum = function (threshold) {
    var ans = 0;
    if (threshold > 0) {
        for(var i = 0; i < this.ny; i=(i+1)|0) {
            for(var h = 0; h < this.nx; h=(h+1)|0) {
                var tmp = this.ary[h][i].life;
                if (tmp >= threshold) {
                    ans = ans+tmp;
                }
            }
        }
    } else {
        for(var i = 0; i < this.ny; i=(i+1)|0) {
            for(var h = 0; h < this.nx; h=(h+1)|0) {
                ans += this.ary[h][i].life;
            }
        }
    }

    return Math.ceil(ans);
}


// グローバル座標上のキャラクター位置を指定位置に移動し、さらに描画領域の中心にオフセット。
// rotateだけだと「左上の原点を中心に、反時計回りで」という直感に反した動きなので注意。
Leaf.prototype.offset = function(ctx, angle, magnify, x, y, W, H, cx, cy) {
    ctx.translate(cx, cy);
    ctx.scale(magnify, magnify); // サイズ変更
    ctx.rotate( -angle ); // 中央に移動し、-rotateして、再び左上に戻る。
    ctx.translate(x-cx, y-cy);
};


// 毎フレーム、キャラクター自身を描画させるメソッド。
Leaf.prototype.drawme = function (ctx) {
    // 定数の初期化
    var LEAF_FILL = "rgba(15, 20, 25, 1)"; // 外枠の塗りつぶし。
    var LEAF_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ctx.darkCol[2]+"%, 1)";

    var LEAF_STROKE = "rgba(0, 0, 5, 1)"; // 外枠の線色。

    ctx.save(); // 事前セーブ

    // グローバル座標上のキャラクター位置を描画領域の中心にオフセットして拡大、回転を掛ける。
    // これを掛けると、後の描画処理はfromX, fromYを基点とするローカル座標上で実施できる。
    this.offset(ctx, this.angle, this.magnify, this._x, this._y, this.W, this.H, this.cx, this.cy);

    var ptrn = ctx.createPattern(this.image, 'repeat');

    // 外枠
    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, this.H);
    ctx.lineTo(this.W, this.H);
    ctx.lineTo(this.W, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
    ctx.lineWidth = 2;
    ctx.strokeStyle = LEAF_STROKE;
    ctx.stroke();
    ctx.fillStyle = LEAF_FILL;
    ctx.fill();

//    console.time('timer_leaf_draw');
    // leaflet の各セルを描画する処理。
    for(var i = 0; i < this.ny; i=(i+1)|0) {
        for(var h = 0; h < this.nx; h=(h+1)|0) {
            this.ary[h][i].drawme(ctx); // このように書ければ楽ですね
        }
    }
//    console.timeEnd('timer_leaf_draw');

    ctx.beginPath();
    ctx.moveTo(0, 0);
    ctx.lineTo(0, this.H);
    ctx.lineTo(this.W, this.H);
    ctx.lineTo(this.W, 0);
    ctx.lineTo(0, 0);
    ctx.closePath();
//    ctx.globalAlpha=0.75;
    ctx.fillStyle = ptrn;
    ctx.fill();

    // 中央マーカー
    ctx.beginPath(); // 新規パスを開始。ここから自機の描画処理。
    ctx.arc(this.W/2, this.H/2, 3, 0, Math.PI * 2, false); // 自機を描くパスを設定
    ctx.fillStyle = "rgb(0,0,0)"; // 自機の色を設定する
//    ctx.fill(); // 自機を描く

    // 事後ロード。これで拡大、オフセット、回転は全てリセットされ、後続の描画処理には影響しない。
    ctx.restore();
}


// 葉っぱの状態を未加害にリセットしてしまうメソッド。
Leaf.prototype.reset = function () {
    for(var i = 0; i < this.ny; i=(i+1)|0) {
        for(var h = 0; h < this.nx; h=(h+1)|0) {
            this.ary[h][i].reset();
        }
    }
}





// ============================================================================


// Leafletクラスを自作する。こっちが、パッチ1枚1枚に相当する。
function Leaflet (where_x, where_y, W, H, angle, magnify) {
    this._x = where_x;
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.life = 1.0; // ここで初期化。あとはフレーム毎に、この値を視て alphaを決める。
    this.angle = angle;
    this.magnify = magnify;
};


// 葉っぱの状態を未加害にリセットしてしまうメソッド。
Leaflet.prototype.reset = function () {
    this.life = 1.0;
};


// 毎フレーム、キャラクター自身を描画させるメソッド。
Leaflet.prototype.drawme = function (ctx) {
//    var LEAFLET_FILL = "rgba(210, 215, 215, " + this.life + ")"; // 葉パッチの色。灰色からスタート
//    var LEAFLET_FILL = "hsla("+(0+this.life*1390)+", 100%, "+ (30+this.life*70) +"%, " + this.life + ")"; // HSL系
//    var LEAFLET_FILL = "hsla("+(-20+this.life*200)+", 10%, "+ (30+this.life*70) +"%, " + this.life + ")"; // HSL系
    var LEAFLET_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ (40+this.life*60) +"%, "+this.life+")";

    var LEAFLET_STROKE = "rgba(255, 255, 255, " + (1-this.life) + ")";
    var MG = 1;
    var RD = 3+(1-this.life)*6 ;

    ctx.save(); // 事前セーブ

    ctx.fillStyle = LEAFLET_FILL;
    ctx.strokeStyle = LEAFLET_STROKE;
    ctx.lineWidth = 3.0;

    // leaflet ポリゴン
    ctx.beginPath();
    ctx.moveTo(this._x       +MG+RD, this._y+MG       );
    ctx.lineTo(this._x+this.W-MG-RD, this._y+MG       );
    ctx.quadraticCurveTo(this._x+this.W-MG, this._y+MG, this._x+this.W-MG, this._y+MG+RD);
    ctx.lineTo(this._x+this.W-MG       , this._y+this.H-MG-RD);
    ctx.quadraticCurveTo(this._x+this.W-MG, this._y+this.H-MG, this._x+this.W-MG-RD, this._y+this.H-MG);
    ctx.lineTo(this._x+MG       +RD, this._y+this.H-MG);
    ctx.quadraticCurveTo(this._x+MG, this._y+this.H-MG, this._x+MG, this._y+this.H-MG-RD);
    ctx.lineTo(this._x+MG              , this._y+MG+RD);
    ctx.quadraticCurveTo(this._x+MG, this._y+MG, this._x+MG+RD, this._y+MG);
    ctx.closePath();
    ctx.fill();

    ctx.restore(); // 事後ロード
}

// 毎フレーム、キャラクター自身を描画させるメソッド。エッジをくっきりさせるため、矩形で描く
Leaflet.prototype.drawLect = function (ctx) {
    var LEAFLET_FILL = "hsla("+(-20+this.life*200)+", 0%, "+ (30+this.life*70) +"%, " + this.life + ")"; // HSL系
    var LEAFLET_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ (40+this.life*60) +"%, "+this.life+")";
    var LEAFLET_STROKE = "rgba(255, 255, 255, " + (1-this.life) + ")";
    var MG = 2;

    ctx.save(); // 事前セーブ

    ctx.fillStyle = LEAFLET_FILL;
    ctx.strokeStyle = LEAFLET_STROKE;
    ctx.lineWidth = 0.5;

    // leaflet 矩形 // 角を落とす処理と比べて、1.5 msくらい速くなる。
    ctx.beginPath();
    ctx.moveTo(this._x+MG       , this._y+MG       );
    ctx.lineTo(this._x+this.W-MG, this._y+MG       );
    ctx.lineTo(this._x+this.W-MG, this._y+this.H-MG);
    ctx.lineTo(this._x+MG       , this._y+this.H-MG);
    ctx.lineTo(this._x+MG       , this._y+MG       );
    ctx.closePath();
    ctx.fill();

    ctx.restore(); // 事後ロード
}



// 毎フレーム、キャラクター自身を描画させるメソッド。円で。
Leaflet.prototype.drawArc = function (ctx) {

    var LEAFLET_FILL = "hsla("+(-20+this.life*200)+", 0%, "+ (30+this.life*70) +"%, " + this.life + ")"; // HSL系
    var LEAFLET_FILL = "hsla("+ctx.liteCol[0]+", "+ctx.liteCol[1]+"%, "+ (40+this.life*60) +"%, "+this.life+")";
    var LEAFLET_STROKE = "rgba(255, 255, 255, " + (1-this.life) + ")";
    var MG = 2;

    ctx.save(); // 事前セーブ

    ctx.fillStyle = LEAFLET_FILL;
    ctx.strokeStyle = LEAFLET_STROKE;
    ctx.lineWidth = 0.5;
    ctx.beginPath(); // 新規パスを開始。ここから自機の描画処理。
    ctx.arc(this._x+this.W/2, this._y+this.H/2, (Math.min(this.W, this.H)/2-MG), 0, Math.PI * 2, false);
    ctx.fill();

    ctx.restore(); // 事後ロード
}
