// ============================================================================
/*
    Copyright 2016 Masaaki Sudo
    http://sudori.info/

    GNU GENERAL PUBLIC LICENSE
    Version 3, 29 June 2007
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    https://www.gnu.org/licenses/gpl.txt
*/
// ============================================================================

// ============================================================================
// グラフ表示のPlotクラス。描画領域、軸、データを内部に含む。
// あまり汎用性の高い関数ではなく、今後の拡張は必要。例えば系列の数を増やすには限界がある。
// よくよく見るとわかるが、下の初期化関数はデータを含まない。setStatus()メソッドで初めてデータを投入する。

// なお、現在のシステムは where_x および whre_y として、「プロットの原点」の位置を指定する。
// マージン部分は Array={bottom, left, top, right} の順に、ピクセル単位で指定する。
// つまり全体の寸法はmargin.left + W + margin.rightといった具合になる。

// 副プロットは ylim_sub, ylab_sub, type_sub として指定
// 副プロットでも複数系列のグラフを表示できるよう、datay_subも「配列の配列」としている。最初は空。
// ============================================================================


function Plot(  where_x, where_y, W, H, margin, xlim, ylim, ylim_sub, xbreaks1, xbreaks2, 
                ybreaks1, ybreaks2, logx, logy, xlab, ylab, ylab_sub, type, type_sub, col_ary, col_sub_ary ) {

    this._x = where_x;
    this._y = where_y;
    this.W = W;
    this.H = H;
    this.xlim = xlim;
    this.ylim = ylim; // Arrayで与える。
    this.ylim_sub = ylim_sub;
    this.logx = logx;
    this.logy = logy; // 主副いずれかだけを対数軸にする機能は今のところ付けていない
    this.margin = margin;
    this.angle = 0;
    this.magnify = 1.0;
    this.type = type; // プロットの種類。現在は "line", "area" のみサポート。
    if (typeof type_sub === "undefined") {
        this.type_sub = type;
    } else {
        this.type_sub = type_sub; // プロットの種類。現在は "line", "area" のみサポート。
    }
    // 初期バージョンではtype だったが、副プロットを入れたのでtype1, type2 に分けた

    this.col_ary = col_ary;
    this.col_sub_ary = col_sub_ary;

    var col=Array(0, 0, 5, 1); // 軸線は黒。
    var font_family='font1';
    var font_size=12;
    var size_unit="px";
    var arrow_scale=0.0;
    this.col = col;
    this.font_family=font_family;
    this.font_size = font_size;
    this.size_unit = size_unit;
    this.arrow_scale = arrow_scale;

    var col_fill=Array(45, 45, 45, 1.0); // 地色は引数に無いので、ひとまず勝手に作っておく。
    var col_border=col; // 軸線に揃えておく
    this.col_fill = col_fill;
    this.col_border = col_border;

    this.xbreaks1 = xbreaks1; // tickmarksは外側で配列を作っているが、あまり良い方法ではない。
    this.xbreaks2 = xbreaks2;
    this.ybreaks1 = ybreaks1;
    this.ybreaks2 = ybreaks2;
    this.xlab = xlab;
    this.ylab = ylab;
    this.ylab_sub = ylab_sub;
    
    // ここから、x軸方向のデータインデックスの作成
    var xmin = xlim[0];
    var xmax = xlim[1];
    this.datax = []; // データ点のインデックスとして、xmin からxmax*5 まで、1ずつ増える数値ベクトルを作成
    for( var i=xmin; this.datax.push(i=(i+1)|0) < Math.floor(xmin+(xmax-xmin)*5););
    // dataxの上限を十分に大きな数字にしないと、xmax よりプレイ時間が増えたときにプロットが上下裏返るバグ発生

    // データ点に対応するx座標は（散布図でなければ）インデックスから計算できるので、予め作っておく
    this.where_datax = this.datax;
    this.where_xmin = this._x;
    this.where_xmax = this._x+this.W;
    if (logx > 0 && xmin > 0 && xmax > xmin) {
        var xmin_log = Math.LOG10E * Math.log(xmin);
        var xmax_log = Math.LOG10E * Math.log(xmax);
        for (var i = 0; i < this.where_datax.length; i=(i+1)|0) {
            this.where_datax[i] = ( this.W*Math.LOG10E*Math.log(this.datax[i]) / (xmax_log-xmin_log) );
        }
    } else {
        for (var i = 0; i < this.where_datax.length; i=(i+1)|0) {
            this.where_datax[i] = ( this.W*this.datax[i] / (xmax-xmin) );
        }
    }
};



// 毎フレーム、プロットの状態を更新するメソッド。
Plot.prototype.setStatus = function(currentFrame, data, data_sub) {
    this.currentFrame = currentFrame; // 現在時間をフレーム数で与える。
    this.datay = data;
    this.datay_sub = data_sub; // なければ、datay_sub に undefined が入る
}


// 毎フレーム、プロットを描画させるメソッド。
Plot.prototype.drawme = function (ctx) {

    ctx.save(); // 事前セーブ

    // プロット位置が移動しうるため、描画関数の中でもx軸の開始地点を再計算するというグランマの生活の知恵
    this.where_xmin = this._x;
    this.where_xmax = this._x+this.W;
    
    // マージンを含めたプロット全体の背景描画。現在は必要ないので透明化している。
    ctx.beginPath();
    ctx.moveTo(this._x - this.margin[1]       , this._y+this.margin[0]);
    ctx.lineTo(this._x+this.W + this.margin[3], this._y+this.margin[0]);
    ctx.lineTo(this._x+this.W + this.margin[3], this._y-this.H-this.margin[2]);
    ctx.lineTo(this._x - this.margin[1]       , this._y-this.H-this.margin[2]);
    ctx.lineTo(this._x - this.margin[1]       , this._y+this.margin[0]);
    ctx.closePath();

    var AREA_STROKE = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0)";
    var AREA_FILL = "hsla("+ctx.baseCol[0]+", "+ctx.baseCol[1]+"%, "+ctx.baseCol[2]+"%, 0)";
    var LINE_WIDTH = 1;
    ctx.strokeStyle = AREA_STROKE;
    ctx.fillStyle = AREA_FILL;
    ctx.lineWidth = LINE_WIDTH;

    ctx.stroke();
    ctx.fill();

    // プロットエリアの背景描画
    var AREA_STROKE = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)";
    var AREA_FILL = "hsla("+ctx.darkCol[0]+", "+ctx.darkCol[1]+"%, "+ctx.darkCol[2]+"%, 1)";
    var LINE_WIDTH = 1.8;

    ctx.strokeStyle = AREA_STROKE;
    ctx.lineWidth = LINE_WIDTH;
    ctx.fillStyle = AREA_FILL;
    ctx.beginPath();
    ctx.moveTo(this._x       , this._y);
    ctx.lineTo(this._x+this.W, this._y);
    ctx.lineTo(this._x+this.W, this._y-this.H);
    ctx.lineTo(this._x       , this._y-this.H);
    ctx.lineTo(this._x       , this._y);
    ctx.closePath();
    ctx.stroke();
    
    ctx.restore(); // 事後ロード

    // 先にsub=0として、主プロットを描画
    if (this.type=="line") {
        this.drawline(ctx, sub=0);
    } else if (this.type=="area") {
        this.drawarea(ctx, sub=0); // ここで、タイトルフレームにおいてundefinedが混入する。
    }
    // 次にsub=1として、副プロットを描画。実は、主が type=="line" の場合、先に副プロットを描いた方が良いかも？
    // コンソールを見る限りデータは間違っていない。が、描画になぜかラグが生じるし、ylimをやっぱり設定する必要がある。
    if (this.type_sub=="line") {
        this.drawline(ctx, sub=1);
    } else if (typeof type_sub !== "undefined" && this.type_sub=="area") {
        this.drawarea(ctx, sub=1);
    }

    ctx.save(); // 事前セーブ

    // ここから軸の描画
    var AXES_STROKE = "rgba("+this.col[0]+", "+this.col[1]+", "+this.col[2]+", "+this.col[3]+")";
    var AXES_FILL = "rgba("+this.col[0]+", "+this.col[1]+", "+this.col[2]+", "+this.col[3]+")";
    var FONT = this.font_size + this.size_unit + " " + this.font_family;
    var LINE_WIDTH = 1;
    
    // x axis
    ctx.beginPath();
    ctx.moveTo(this._x       , this._y);
    ctx.lineTo(this._x+this.W, this._y);
    ctx.closePath();
    ctx.strokeStyle = AXES_STROKE;
    ctx.lineWidth = LINE_WIDTH;
    ctx.stroke();

    // x axis: arrows
    ctx.lineWidth = LINE_WIDTH;
    ctx.beginPath();
    ctx.moveTo(this._x+this.W, this._y);
    ctx.lineTo(this._x+this.W-(this.W*this.arrow_scale*0.075), this._y-(this.W*this.arrow_scale*0.035));
    ctx.closePath();
    ctx.stroke();
    ctx.beginPath();
    ctx.moveTo(this._x+this.W, this._y);
    ctx.lineTo(this._x+this.W-(this.W*this.arrow_scale*0.075), this._y+(this.W*this.arrow_scale*0.035));
    ctx.closePath();
    ctx.stroke();

    // y axis
    ctx.beginPath();
    ctx.moveTo(this._x       , this._y);
    ctx.lineTo(this._x, this._y-this.H);
    ctx.closePath();
    ctx.strokeStyle = AXES_STROKE;
    ctx.lineWidth = LINE_WIDTH;
    ctx.stroke();
    
    // y axis: arrows
    ctx.lineWidth = LINE_WIDTH;
    ctx.beginPath();
    ctx.moveTo(this._x, this._y-this.H);
    ctx.lineTo(this._x+(this.W*this.arrow_scale*0.035), this._y-this.H+(this.W*this.arrow_scale*0.075));
    ctx.closePath();
    ctx.stroke();
    ctx.beginPath();
    ctx.moveTo(this._x, this._y-this.H);
    ctx.lineTo(this._x-(this.W*this.arrow_scale*0.035), this._y-this.H+(this.W*this.arrow_scale*0.075));
    ctx.closePath();
    ctx.stroke();

    // y axis sub
    ctx.beginPath();
    ctx.moveTo(this._x+this.W       , this._y);
    ctx.lineTo(this._x+this.W, this._y-this.H);
    ctx.closePath();
    ctx.strokeStyle = AXES_STROKE;
    ctx.lineWidth = LINE_WIDTH;
    ctx.stroke();
    
    // y axis sub: arrows
    ctx.lineWidth = LINE_WIDTH;
    ctx.beginPath();
    ctx.moveTo(this._x+this.W, this._y-this.H);
    ctx.lineTo(this._x+this.W+(this.W*this.arrow_scale*0.035), this._y-this.H+(this.W*this.arrow_scale*0.075));
    ctx.closePath();
    ctx.stroke();
    ctx.beginPath();
    ctx.moveTo(this._x+this.W, this._y-this.H);
    ctx.lineTo(this._x+this.W-(this.W*this.arrow_scale*0.035), this._y-this.H+(this.W*this.arrow_scale*0.075));
    ctx.closePath();
    ctx.stroke();

    // x tickmarks なお、xbreaks が配列形式のデータである事が前提。
    var xticks_len = this.xbreaks1.length;
    var xmin = this.xlim[0];
    var xmax = this.xlim[1];
    var where_xbreaks1 = Array(xticks_len);
    if (logx > 0 && xmin > 0 && xmax > 0) {
        xmin_log = Math.LOG10E * Math.log(xmin);
        xmax_log = Math.LOG10E * Math.log(xmax);
        for (var i = 0; i < xticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.xbreaks1[i]);
            where_xbreaks1[i] = this.where_xmin + this.W * d_log / (xmax_log-xmin_log);
        }
    } else {
        for (var i = 0; i < xticks_len; i=(i+1)|0) {
            where_xbreaks1[i] = this.where_xmin + this.W*this.xbreaks1[i]/(xmax-xmin);
        }
    }
    ctx.lineWidth = LINE_WIDTH;
    ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.xbreaks1.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(where_xbreaks1[i], this._y);
        ctx.lineTo(where_xbreaks1[i], this._y+2); // lengths of thickmarks
        ctx.closePath();
        ctx.stroke();
    }

    // x tickmarks 2
    var xticks_len = this.xbreaks2.length;
    var xmin = this.xlim[0];
    var xmax = this.xlim[1];
    var where_xbreaks2 = Array(xticks_len);
    if (logx > 0 && xmin > 0 && xmax > 0) {
        var xmin_log = Math.LOG10E * Math.log(xmin);
        var xmax_log = Math.LOG10E * Math.log(xmax);
        for (var i = 0; i < xticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.xbreaks2[i]);
            where_xbreaks2[i] = this.where_xmin + this.W * d_log / (xmax_log-xmin_log);
        }
    } else {
        for (var i = 0; i < xticks_len; i=(i+1)|0) {
            where_xbreaks2[i] = this.where_xmin + this.W*this.xbreaks2[i]/(xmax-xmin);
        }
    }
    ctx.lineWidth = LINE_WIDTH;
    ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.xbreaks2.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(where_xbreaks2[i], this._y);
        ctx.lineTo(where_xbreaks2[i], this._y+5); // lengths of thickmarks
        ctx.closePath();
        ctx.stroke();
    }

    // y tickmarks なお、ybreaks が配列形式のデータである事が前提。
    var yticks_len = this.ybreaks1.length;
    var ymin = this.ylim[0];
    var ymax = this.ylim[1];
    var where_ymin = this._y;
    var where_ymax = this._y-this.H;
    var where_ybreaks1 = Array(yticks_len);
    if (logy > 0 && ymin > 0 && ymax > 0) {
        var ymin_log = Math.LOG10E * Math.log(ymin);
        var ymax_log = Math.LOG10E * Math.log(ymax);
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.ybreaks1[i]);
            where_ybreaks1[i] = where_ymin - this.H * d_log / (ymax_log-ymin_log);
        }
    } else {
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            where_ybreaks1[i] = where_ymin - this.H*this.ybreaks1[i]/(ymax-ymin);
        }
    }
        ctx.lineWidth = LINE_WIDTH;
        ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.ybreaks1.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(this._x, where_ybreaks1[i]);
        ctx.lineTo(this._x-2, where_ybreaks1[i]);
        ctx.closePath();
        ctx.stroke();
    }

    // y tickmarks 2
    var yticks_len = this.ybreaks2.length;
    var ymin = this.ylim[0];
    var ymax = this.ylim[1];
    var where_ymin = this._y;
    var where_ymax = this._y-this.H;
    var where_ybreaks2 = Array(yticks_len);
    if (logy > 0 && ymin > 0 && ymax > 0) {
        var ymin_log = Math.LOG10E * Math.log(ymin);
        var ymax_log = Math.LOG10E * Math.log(ymax);
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.ybreaks2[i]);
            where_ybreaks2[i] = where_ymin - this.H * d_log / (ymax_log-ymin_log);
        }
    } else {
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            where_ybreaks2[i] = where_ymin - this.H*this.ybreaks2[i]/(ymax-ymin);
        }
    }
        ctx.lineWidth = LINE_WIDTH;
        ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.ybreaks2.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(this._x, where_ybreaks2[i]);
        ctx.lineTo(this._x-5, where_ybreaks2[i]);
        ctx.closePath();
        ctx.stroke();
    }

    // y tickmarks sub なお、 ybreaks が配列形式のデータである事が前提。
    var yticks_len = this.ybreaks1.length;
    var ymin = this.ylim[0];
    var ymax = this.ylim[1];
    var where_ymin = this._y;
    var where_ymax = this._y-this.H;
    var where_ybreaks1 = Array(yticks_len);
    if (logy > 0 && ymin > 0 && ymax > 0) {
        var ymin_log = Math.LOG10E * Math.log(ymin);
        var ymax_log = Math.LOG10E * Math.log(ymax);
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.ybreaks1[i]);
            where_ybreaks1[i] = where_ymin - this.H * d_log / (ymax_log-ymin_log);
        }
    } else {
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            where_ybreaks1[i] = where_ymin - this.H*this.ybreaks1[i]/(ymax-ymin);
        }
    }
    ctx.lineWidth = LINE_WIDTH;
    ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.ybreaks1.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(this._x+this.W, where_ybreaks1[i]);
        ctx.lineTo(this._x+this.W+2, where_ybreaks1[i]);
        ctx.closePath();
        ctx.stroke();
    }

    // y tickmarks sub 2
    var yticks_len = this.ybreaks2.length;
    var ymin = this.ylim[0];
    var ymax = this.ylim[1];
    var where_ymin = this._y;
    var where_ymax = this._y-this.H;
    var where_ybreaks2 = Array(yticks_len);
    if (logy > 0 && ymin > 0 && ymax > 0) {
        var ymin_log = Math.LOG10E * Math.log(ymin);
        var ymax_log = Math.LOG10E * Math.log(ymax);
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            var d_log = Math.LOG10E * Math.log(this.ybreaks2[i]);
            where_ybreaks2[i] = where_ymin - this.H * d_log / (ymax_log-ymin_log);
        }
    } else {
        for (var i = 0; i < yticks_len; i=(i+1)|0) {
            where_ybreaks2[i] = where_ymin - this.H*this.ybreaks2[i]/(ymax-ymin);
        }
    }
    ctx.lineWidth = LINE_WIDTH;
    ctx.strokeStyle = AXES_STROKE;
    for (var i=0; i<this.ybreaks2.length; i=(i+1)|0) {
        ctx.beginPath();
        ctx.moveTo(this._x+this.W, where_ybreaks2[i]);
        ctx.lineTo(this._x+this.W+5, where_ybreaks2[i]);
        ctx.closePath();
        ctx.stroke();
    }

    // labels
    ctx.textAlign = "center";
    ctx.fillStyle = AXES_FILL;
    ctx.font = FONT;
    
    ctx.fillText(this.xlab, this._x + this.W/2, this._y+this.font_size*1.5 ); // x label
    ctx.translate( this._x , this._y - this.H/2 );
    ctx.rotate(-Math.PI/2);
    ctx.translate( -this._x, -this._y+this.H/2 );
    ctx.fillText(this.ylab, this._x-0*this.font_size*1.5 , this._y - this.H/2 - this.font_size*0.75 ); // y label
    ctx.translate( 0 , this.W );
    ctx.fillText(this.ylab_sub, this._x-0*this.font_size*1.5 , this._y - this.H/2 + this.font_size*1.4 ); // y label sub
    
    ctx.restore(); // 事後ロード
}


// type=="line" であるときに、折れ線グラフの実体を描画させるメソッド

Plot.prototype.drawline = function (ctx, sub) {

    if (sub <= 0) {
        var PLOT_COLOR_ARRAY = this.col_ary;
        var PLOT_WIDTH = 1.0;
        var dy = this.datay;
        var yl = this.ylim;
    } else {
        var PLOT_COLOR_ARRAY = this.col_sub_ary;
        var PLOT_WIDTH = 1.5;
        var dy = this.datay_sub;
        var yl = this.ylim_sub;
    }
    
    ctx.save(); // 事前セーブ

    // i=0 と i=6 はそれぞれ、未割当、死亡なのでいらない
    for (var i = 1; i < dy.length-1; i=(i+1)|0) {
        var ymin = yl[0];
        var ymax = yl[1];
        var where_ymin = this._y;
        var where_ymax = this._y-this.H;
        var where_datay = Array(dy[i].length);
        if (logy > 0 && ymin > 0 && ymax > 0) {
            var ymin_log = Math.LOG10E * Math.log(ymin);
            var ymax_log = Math.LOG10E * Math.log(ymax);
            for (var j = 0; j < dy[i].length; j=(j+1)|0) {
                where_datay[j] = where_ymin - ( this.H*Math.LOG10E*Math.log(dy[i][j]) / (ymax_log-ymin_log) );
            }
        } else {
            for (var j = 0; j < dy[i].length; j=(j+1)|0) {
                where_datay[j] = where_ymin - ( this.H*dy[i][j]/(ymax-ymin) );
            }
        }

        if (sub > 0) {
            ctx.beginPath();
            ctx.moveTo(this._x+this.where_datax[0], where_datay[0]);
            for (var j = 1; j < where_datay.length; j=(j+1)|0) {
                ctx.lineTo(this._x+this.where_datax[j], where_datay[j]);
            }
            ctx.lineWidth = PLOT_WIDTH+0.75;
            ctx.strokeStyle = "rgba(0, 0, 0, 1)";
            ctx.stroke(); // strokeで輪郭。subだと別のグラフに被せて表示するので、先に縁取りを作る。
        }
        ctx.beginPath();
        ctx.moveTo(this._x+this.where_datax[0], where_datay[0]);
        for (var j = 1; j < where_datay.length; j=(j+1)|0) {
            ctx.lineTo(this._x+this.where_datax[j], where_datay[j]);
        }
        ctx.lineWidth = PLOT_WIDTH;
        ctx.fillStyle = PLOT_COLOR_ARRAY[i-1];
        ctx.strokeStyle = PLOT_COLOR_ARRAY[i-1];
        ctx.stroke(); // strokeで輪郭。
    }

    ctx.restore(); // 事後ロード
}



// type=="area" であるときに、積み上げ折れ線グラフ（エリアプロット）の実体を描画させるメソッド
// 問題は、最初のカテゴリから当該カテゴリまでを含む、
// これは最初からデータを用意させるのが妥当。ruisekiと称して配列データを作っておき、setStatusの際に、該当データ部分を計算させる。
// ただしデータはPlotのオブジェクト内に保持するのではなく、あらかじめ他所で作成。

// あと描画時のポリゴンの扱いも問題で、上面はデータそのものの高さとして下面を「前のデータの高さ」にするか、「y=0」にするか。
// 前者は計算が簡単だしおそらく処理速度も速いが、エリアが半透明である際に被り部分の色表示がおかしくなる。
// 一つの解決法はグローバルアルファを、drawarea関数に入っている間だけ変更する。個別のカテゴリに対するアルファは指定出来ないが、
// おそらくそれが問題になる局面はほとんどないので。
// まあ、両方実装してみて早い方に決めればイインジャナイかな？


Plot.prototype.drawarea = function (ctx, sub) {

    var PLOT_STROKE = "rgba(5, 5, 5, 1)";
    if (sub <= 0) {
        var PLOT_COLOR_ARRAY = this.col_ary; // 現在のカラー定義は系列数5を想定した手抜き工事なので注意。
        var PLOT_WIDTH = 0.5;
        var dy = this.datay;
        var yl = this.ylim;
    } else {
        var PLOT_COLOR_ARRAY = this.col_sub_ary;
        var PLOT_WIDTH = 1.0;
        var dy = this.datay_sub;
        var yl = this.ylim_sub;
    }

    ctx.save(); // 事前セーブ

    // i=4 から i=0 まで、デクリメントしながら5回描画する。
    for (var i = dy.length-1; i >= 0 ; i=(i-1)|0) {
        var ymin = yl[0];
        var ymax = yl[1];
        var where_ymin = this._y;
        var where_ymax = this._y-this.H;
        var where_datay = Array(this.currentFrame);
        // logyオプションが指定された場合、10を底とする対数に変換。ただし積み上げでlogは、そもそもおかしい
        if (logy > 0 && ymin > 0 && ymax > 0) {
            var ymin_log = Math.LOG10E * Math.log(ymin);
            var ymax_log = Math.LOG10E * Math.log(ymax);
            for (var j = 0; j < this.currentFrame; j=(j+1)|0) {
                where_datay[j] = where_ymin - ( this.H*Math.LOG10E*Math.log(dy[i][j]) / (ymax_log-ymin_log) );
            }
        } else {
            for (var j = 0; j < this.currentFrame; j=(j+1)|0) {
                where_datay[j] = where_ymin - ( this.H*dy[i][j]/(ymax-ymin) );
            }
        }

        ctx.beginPath();
        ctx.moveTo(this._x+this.where_datax[0], this._y); // 原点
        ctx.lineTo(this._x+this.where_datax[0], where_datay[0]); // 最初のデータ点
        for (var j = 1; j < this.currentFrame; j=(j+1)|0) {
            ctx.lineTo(this._x+this.where_datax[j], where_datay[j]);
        }
        ctx.lineTo(this._x+this.where_datax[this.currentFrame-1], this._y); // 右下
        ctx.lineTo(this._x+this.where_datax[0], this._y); // 原点に戻る
        ctx.closePath();

        ctx.lineJoin = "round";
        ctx.lineWidth = PLOT_WIDTH;
        ctx.strokeStyle = PLOT_STROKE;
        ctx.stroke(); // strokeで輪郭。
        ctx.fillStyle = PLOT_COLOR_ARRAY[i];
        ctx.fill();
    }
    ctx.restore(); // 事後ロード
}


